<?php
namespace App\Controllers;
use ORM;
use PDO;
use App\Controllers\BaseController;
use App\Models\User;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Item;
use App\Models\Project;
use App\Models\Contact;
use App\Models\Transaction;
use Carbon\Carbon;

require_once __DIR__ . '/../util/global.php';

class DashboardController extends BaseController
{
    public function getDashboardData($all_data_access = false)
    {
        header('Content-Type: application/json');

        // Authenticated user has all-data access?
        $user = User::_info();
        $GLOBALS['user'] = $user;
        $hasAllDataPermission = _hasaccess($user->roleid, 'transactions', 'all_data');

        $all_data_access = $hasAllDataPermission;

        $response = [];
        $incomeVsExpStart = $_GET['start_date'] ?? null;

        // Rango dinámico: weekly|monthly|yearly|custom
        $rangeParam = strtolower($_GET['range'] ?? 'weekly');
        $customStart = $_GET['start_date'] ?? null;
        $customEnd = $_GET['end_date'] ?? null;

        $now = Carbon::now();
        switch ($rangeParam) {
            case 'monthly':
                $rangeStart = $now->copy()->startOfMonth();
                $rangeEnd = $now->copy()->endOfMonth();
                break;
            case 'yearly':
                $rangeStart = $now->copy()->startOfYear();
                $rangeEnd = $now->copy()->endOfYear();
                break;
            case 'custom':
                try {
                    $rangeStart = Carbon::parse($customStart);
                    $rangeEnd = $customEnd ? Carbon::parse($customEnd) : Carbon::parse($customStart);
                    if ($rangeEnd->lessThan($rangeStart)) {
                        [$rangeStart, $rangeEnd] = [$rangeEnd, $rangeStart];
                    }
                } catch (\Throwable $e) {
                    $rangeStart = $now->copy()->startOfWeek(Carbon::MONDAY);
                    $rangeEnd = $now->copy()->startOfWeek(Carbon::MONDAY)->addDays(5);
                }
                break;
            case 'weekly':
            default:
                $rangeStart = $now->copy()->startOfWeek(Carbon::MONDAY);
                $rangeEnd = $now->copy()->startOfWeek(Carbon::MONDAY)->addDays(5);
                break;
        }

        $rangeDays = $rangeStart->diffInDays($rangeEnd) + 1;
        $prevRangeEnd = $rangeStart->copy()->subDay();
        $prevRangeStart = $prevRangeEnd->copy()->subDays($rangeDays - 1);

        $rangeStartDate = $rangeStart->toDateString();
        $rangeEndDate = $rangeEnd->toDateString();
        $prevRangeStartDate = $prevRangeStart->toDateString();
        $prevRangeEndDate = $prevRangeEnd->toDateString();

        $net_worth = $this->getNetWorthRange($rangeStartDate, $rangeEndDate, $all_data_access, $user);

        try {
            $income_vs_exp = $this->getIncomeExpenseSeries($rangeStartDate, $rangeEndDate);
        } catch (\Throwable $e) {
            $income_vs_exp = ['Months' => [], 'Income' => [], 'Expense' => []];
        }

        $payment_methods_comparison = $this->getPaymentMethodsComparison($rangeStartDate, $rangeEndDate);
        $metrics_invoices365 = $this->getMetricsInvoicesRange($rangeStartDate, $rangeEndDate);
        $recent_sales = $this->getRecentSales($rangeStartDate, $rangeEndDate);
        $top_services = $this->computeTopServicesChartData(null, 5, $rangeStartDate, $rangeEndDate);
        $staff_sales_week = $this->getStaffSalesRange($rangeStartDate . ' 00:00:00', $rangeEndDate . ' 23:59:59', $all_data_access, $user);

        if ($all_data_access) {

            // Totales en rango actual y previo
            $total_sales = ORM::for_table('sys_invoices')
                ->where_gte('date', $rangeStartDate)
                ->where_lte('date', $rangeEndDate)
                ->sum('total') ?: '0.00';
            $total_sales_prev = ORM::for_table('sys_invoices')
                ->where_gte('date', $prevRangeStartDate)
                ->where_lte('date', $prevRangeEndDate)
                ->sum('total') ?: '0.00';

            // Total Invoices
            $total_invoices_count = ORM::for_table('sys_invoices')
                ->where_gte('date', $rangeStartDate)
                ->where_lte('date', $rangeEndDate)
                ->count() ?: 0;
            $total_invoices_prev = ORM::for_table('sys_invoices')
                ->where_gte('date', $prevRangeStartDate)
                ->where_lte('date', $prevRangeEndDate)
                ->count() ?: 0;

            // Paid / Partially Paid / Pending
            $paid_invoices_count = ORM::for_table('sys_invoices')
                ->where('status', 'Paid')
                ->where_gte('date', $rangeStartDate)
                ->where_lte('date', $rangeEndDate)
                ->count() ?: 0;
            $paid_invoices_prev = ORM::for_table('sys_invoices')
                ->where('status', 'Paid')
                ->where_gte('date', $prevRangeStartDate)
                ->where_lte('date', $prevRangeEndDate)
                ->count() ?: 0;

            $partially_paid_invoices_count = ORM::for_table('sys_invoices')
                ->where('status', 'Partially Paid')
                ->where_gte('date', $rangeStartDate)
                ->where_lte('date', $rangeEndDate)
                ->count() ?: 0;
            $partially_paid_prev = ORM::for_table('sys_invoices')
                ->where('status', 'Partially Paid')
                ->where_gte('date', $prevRangeStartDate)
                ->where_lte('date', $prevRangeEndDate)
                ->count() ?: 0;

            $pending_invoices_count = ORM::for_table('sys_invoices')
                ->where('status', 'Unpaid')
                ->where_gte('date', $rangeStartDate)
                ->where_lte('date', $rangeEndDate)
                ->count() ?: 0;

            $paid_invoices_amount = ORM::for_table('sys_invoices')
                ->where('status', 'Paid')
                ->where_gte('date', $rangeStartDate)
                ->where_lte('date', $rangeEndDate)
                ->sum('total') ?: '0.00';

            $pending_invoices_amount = ORM::for_table('sys_invoices')
                ->where('status', 'Unpaid')
                ->where_gte('date', $rangeStartDate)
                ->where_lte('date', $rangeEndDate)
                ->sum('total') ?: '0.00';

            

            $response = [
                'total_sales'    => $total_sales,
                'total_invoices_count' => $total_invoices_count,
                'paid_invoices_count' => $paid_invoices_count,
                'partially_paid_invoices_count' => $partially_paid_invoices_count,
                'pending_invoices_count' => $pending_invoices_count,
                'paid_invoices_amount' => $paid_invoices_amount,
                'pending_invoices_amount' => $pending_invoices_amount,
                'payment_methods_comparison' => $payment_methods_comparison,
                'metrics_invoices365' => $metrics_invoices365,
                'recent_sales' => $recent_sales,
                'top_services' => $top_services,
                'income_vs_exp' => $income_vs_exp,
                'staff_sales_week' => $staff_sales_week,
                'net_worth' => $net_worth,
                'trending' => [
                    'sales_pct' => $this->percentageChange($total_sales, $total_sales_prev),
                    'invoices_pct' => $this->percentageChange($total_invoices_count, $total_invoices_prev),
                    'paid_partial_pct' => $this->percentageChange(
                        ($paid_invoices_count + $partially_paid_invoices_count),
                        ($paid_invoices_prev + $partially_paid_prev)
                    ),
                ],
            ];

        } else {

            // Últimos 30 días
            $from30 = Carbon::now()->subDays(30)->toDateString();
            $today  = Carbon::now()->toDateString();

            $customers_total = Contact::where('type', 'like', '%Customer%')
                ->whereBetween('created_at', [$from30, $today])
                ->count() ?: '0';

            $companies_total = ORM::for_table('sys_companies')
                ->where_gte('created_at', $from30)
                ->where_lte('created_at', $today)
                ->count() ?: '0';

            $leads_total = ORM::for_table('crm_leads')
                ->where_gte('created_at', $from30)
                ->where_lte('created_at', $today)
                ->count() ?: '0';

            $response = [
                'customers_total' => $customers_total,
                'companies_total' => $companies_total,
                'leads_total' => $leads_total,
                'payment_methods_comparison' => $payment_methods_comparison,
                'metrics_invoices365' => $metrics_invoices365,
                'recent_sales' => $recent_sales,
                'top_services' => $top_services,
                'income_vs_exp' => $income_vs_exp,
                'staff_sales_week' => $staff_sales_week,
                'net_worth' => $net_worth,
            ];
            
        }

        http_response_code(200);
        echo json_encode($response);
        exit;

    }

    private function dataLastTwelveMonthsIncExp()
    {
        global $user, $_L;

        $all_data = _hasaccess($user->roleid, 'transactions', 'all_data');

        // Rango de fecha
        $start_date = date('Y-m-01', strtotime('-11 months'));
        $end_date = date('Y-m-t');

        // Base query
        $query = ORM::for_table('sys_transactions')
            ->select_expr("YEAR(date) AS y")
            ->select_expr("MONTH(date) AS m")
            ->select('type')
            ->select_expr("SUM(amount * currency_rate) AS total")
            ->where_gte('date', $start_date)
            ->where_lte('date', $end_date);

        if (!$all_data) {
            $query->where('aid', $user->id);
        }

        $query->group_by_expr('YEAR(date), MONTH(date), type');

        $results = $query->find_array();

        // Inicializar meses
        $months = [];
        $income = [];
        $expense = [];
        $translations = (isset($_L) && is_array($_L)) ? $_L : [];

        for ($i = 11; $i >= 0; $i--) {
            $timestamp = strtotime("-$i months", strtotime(date('Y-m-01')));
            $months_key = date('Y-m', $timestamp);
            $monthName = date('M', $timestamp);
            $label = $translations[$monthName] ?? $monthName;
            $months[] = $label . ' ' . date('Y', $timestamp);
            $income[$months_key] = '0.00';
            $expense[$months_key] = '0.00';
        }

        // Mapear resultados
        foreach ($results as $row) {
            $key = sprintf('%04d-%02d', $row['y'], $row['m']);
            if ($row['type'] === 'Income') {
                $income[$key] = round((float) $row['total']);
            } elseif ($row['type'] === 'Expense') {
                $expense[$key] = round((float) $row['total']);
            }
        }

        // Formatear respuesta
        $inc_array = [];
        $exp_array = [];

        foreach (array_keys($income) as $k) {
            $inc_array[] = $income[$k];
            $exp_array[] = $expense[$k];
        }

        return [
            'Months' => $months,
            'Income' => $inc_array,
            'Expense' => $exp_array,
        ];
    }

    private function dataLastTwelveMonthsIncExpByFecha($fecha_inicio)
    {

        $month_start_date = date('Y-m-01', strtotime($fecha_inicio));
        $month_end_date = date('Y-m-t', strtotime($fecha_inicio));

        global $user, $_L;

        $all_data = true;
        if (!_hasaccess($user->roleid, 'transactions', 'all_data')) {
            $all_data = false;
        }

        $months = [];
        $translations = (isset($_L) && is_array($_L)) ? $_L : [];

        for ($i = 1; $i <= 11; $i++) {
            //$months[] = date("M Y", strtotime(date('Y-m-01') . " -$i months"));
            $months[] = date("M Y", strtotime(date('Y-m-01', strtotime($fecha_inicio)) . " -$i months"));
        }

        $months = array_reverse($months);

        $months[12] = date("M Y", strtotime(date('Y-m-01', strtotime($fecha_inicio))));
        //$months[12] = date("M Y", strtotime(date('Y-m-01')));

        $inc = [];
        $exp = [];
        $m = [];

        foreach ($months as $month) {
            $d_array = explode(' ', $month);

            $m_short = '';

            if (isset($d_array['0'])) {
                $m_short = $d_array[0];
            }

            if (isset($translations[$m_short])) {
                $m_short = $translations[$m_short];
            }

            $y = '';
            if (isset($d_array[1])) {
                $y = $d_array[1];
            }

            $m[] = $m_short . ' ' . $y;
        }

        $i = 0;

        foreach ($months as $month) {
            $first_day_this_month = date(
                "Y-m-d",
                strtotime("first day of $month")
            );
            $last_day_this_month = date(
                "Y-m-d",
                strtotime("last day of $month")
            );

            $total = 0;

            if (true)
            {
                $transactions = new Transaction();

                $transactions = $transactions->where('type', 'Income')
                    ->where(function ($query) use ($first_day_this_month, $last_day_this_month) {
                        $query->where('date', '>=', $first_day_this_month)
                            ->where('date', '<=', $last_day_this_month);
                    });

                if (!$all_data) {
                    $transactions = $transactions->where('aid', $user->id);
                }

                $transactions = $transactions->get();

                foreach ($transactions as $transaction) {
                    $total += $transaction->amount * $transaction->currency_rate;
                }

                $inc[] = $total;

                $inc[$i] = $inc[$i] == '' ? '0.00' : round($inc[$i]);
            } else {
                $inc[] = 0;
            }

            $i++;
        }

        $i = 0;

        $xd = [];

        foreach ($months as $month) {

            $first_day_this_month = date(
                "Y-m-d",
                strtotime("first day of $month")
            );
            $last_day_this_month = date(
                "Y-m-d",
                strtotime("last day of $month")
            );

            $total = 0;

            if (true)
            //if((strtotime($first_day_this_month) == strtotime($month_start_date)) && (strtotime($last_day_this_month) == strtotime($month_end_date)) )
            {
                $transactions = new Transaction();

                $transactions = $transactions->where('type', 'Expense')->where(function ($query) use ($first_day_this_month, $last_day_this_month) {
                    $query->where('date', '>=', $first_day_this_month)
                        ->where('date', '<=', $last_day_this_month);
                });

                if (!$all_data) {
                    $transactions = $transactions->where('aid', $user->id);
                }

                $transactions = $transactions->get();

                foreach ($transactions as $transaction) {
                    $total += $transaction->amount * $transaction->currency_rate;
                }

                $exp[] = $total;

                $exp[$i] = $exp[$i] == '' ? '0.00' : round($exp[$i]);
            } else {
                $exp[] = 0;
            }

            $xd[] = array(
                "inicio_mes" => $first_day_this_month,
                "fin_mes" => $last_day_this_month,
                "es igual" => (strtotime($first_day_this_month) == strtotime($month_start_date)) && (strtotime($last_day_this_month) == strtotime($month_end_date)) ? "es igual" : "no es igual",
                "inicio_mes_comparar" => $month_start_date,
                "fin_mes_comparar" => $month_end_date,
            );

            $i++;
        }

        return [
            'Months' => $m,
            'Income' => $inc,
            'Expense' => $exp,
            'xd' => $xd,
        ];
    }

    private function getPaymentMethodsComparison($fromDate = null, $toDate = null)
    {
        /*
        * Se lee de sys_transactions con type = 'Income' o type 'In' o 'Out'
        * Agrupa por metodo de pago (method),
        */
        $payment_methods_query = "
            SELECT 
                st.method AS payment_type,
                SUM(st.amount) AS total_amount
            FROM sys_transactions st
            WHERE (st.type = 'Income' OR st.type IN ('In','Out'))
                AND st.date BETWEEN :fromDate AND :toDate
        ";

        $payment_methods_query .= " GROUP BY payment_type ORDER BY total_amount DESC;";

        $pdo = ORM::get_db();
        $stmt = $pdo->prepare($payment_methods_query);

        if ($fromDate && $toDate) {
            $stmt->bindValue(':fromDate', $fromDate);
            $stmt->bindValue(':toDate', $toDate);
        }

        $stmt->execute();

        $payment_methods_comparison = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return $payment_methods_comparison;
    }

    private function getMetricsInvoicesRange($weekStartDate, $weekEndDate)
    {
        // Total issued semana en curso (all invoices, any status)
        $invTotal7 = Invoice::whereBetween('date', [$weekStartDate, $weekEndDate])
            ->sum('total') ?? 0;

        $countTotal = Invoice::whereBetween('date', [$weekStartDate, $weekEndDate])
            ->count() ?? 0;

        // Unpaid semana en curso y desglose Overdue / Not due yet
        $invUnpaid7 = Invoice::where('status', 'Unpaid')
         ->whereBetween('date', [$weekStartDate, $weekEndDate])
         ->sum('total') ?? 0;

        $invOverdue7 = Invoice::where('status', 'Unpaid')
            ->whereBetween('date', [$weekStartDate, $weekEndDate])
            ->where('duedate', '<=', $weekEndDate)
            ->sum('total') ?? 0;

        $invNotDue7 = max(0, $invUnpaid7 - $invOverdue7);

        // Paid semana en curso
        $invPaid7 = Invoice::where('status', 'Paid')
            ->whereBetween('date', [$weekStartDate, $weekEndDate])
            ->sum('total') ?? 0;

        // Partially Paid semana en curso
        $invPartiallyPaid7 = Invoice::where('status', 'Partially Paid')
            ->whereBetween('date', [$weekStartDate, $weekEndDate])
            ->sum('total') ?? 0;

        // Conteos
        $countPaid = Invoice::where('status', 'Paid')
            ->whereBetween('date', [$weekStartDate, $weekEndDate])
            ->count() ?? 0;

        $countPartially = Invoice::where('status', 'Partially Paid')
            ->whereBetween('date', [$weekStartDate, $weekEndDate])
            ->count() ?? 0;

        $countUnpaid = Invoice::where('status', 'Unpaid')
            ->whereBetween('date', [$weekStartDate, $weekEndDate])
            ->count() ?? 0;

        // "Deposited" aproximado: Income vinculados a factura (iid > 0) semana en curso
        $txDeposited7 = ORM::for_table('sys_transactions')
            ->where('type', 'Income')
            ->where_gte('date', $weekStartDate)
            ->where_lte('date', $weekEndDate)
            ->where_gt('iid', 0)
            ->sum('amount') ?: 0;

        $txNotDeposited7 = max(0, $invPaid7 - $txDeposited7);

        // Depósitos Stripe (Income cuyo método contenga stripe)
        $stripeGross = ORM::for_table('sys_transactions')
            ->where('type', 'Income')
            ->where_gt('iid', 0)
            ->where_gte('date', $weekStartDate)
            ->where_lte('date', $weekEndDate)
            ->where_raw('LOWER(account) IN (?, ?)', ['stripe hn', 'stripe va'])
            ->sum('amount') ?: 0;
        // Stripe deposits netos menos 3% de comisión
        $stripeDeposits7 = (float)$stripeGross * 0.97;

        // Total depositado en banco (todas las entradas)
        $bankDeposits7 = ORM::for_table('sys_transactions')
            ->where('type', 'Income')
            ->where_gt('iid', 0)
            ->where_gte('date', $weekStartDate)
            ->where_lte('date', $weekEndDate)
            ->sum('amount') ?: 0;

        $soldAmountPaidPartial = $invPaid7 + $invPartiallyPaid7;

        return [
            // Nuevas metricas semanales
            'sold_total' => $invTotal7,
            'stripe_deposits_total' => $stripeDeposits7,
            'invoice_total_paid_partial' => $soldAmountPaidPartial,
            'bank_deposits_total' => $bankDeposits7,
            'count_sales' => $countTotal,
            'count_paid' => $countPaid,
            'count_partial' => $countPartially,
            'count_unpaid' => $countUnpaid,

            // Metricas ultimos 7 dias
            'inv_unpaid_365_total' => $invUnpaid7,
            'inv_overdue_365_total' => $invOverdue7,
            'inv_not_due_365_total' => $invNotDue7,
            'inv_paid_30_total' => $invPaid7,
            'inv_partially_paid' => $invPartiallyPaid7,
            'inv_deposited_30_total' => $txDeposited7,
            'inv_not_deposited_30_total' => $txNotDeposited7
        ];
    }

    private function getIncomeExpenseSeries($fromDate, $toDate)
    {
        global $user;

        $all_data = _hasaccess($user->roleid, 'transactions', 'all_data');

        $start = Carbon::parse($fromDate);
        $end = Carbon::parse($toDate);
        $diffDays = $start->diffInDays($end);

        $groupDaily = $diffDays <= 45;

        $query = ORM::for_table('sys_transactions')
            ->select('type');

        if ($groupDaily) {
            $query->select_expr("DATE(date) AS period");
            $groupExpr = 'DATE(date)';
        } else {
            $query->select_expr("DATE_FORMAT(date, '%Y-%m-01') AS period");
            $groupExpr = "DATE_FORMAT(date, '%Y-%m-01')";
        }

        $query->select_expr("SUM(amount * currency_rate) AS total")
            ->where_in('type', ['Income', 'Expense'])
            ->where_gte('date', $fromDate)
            ->where_lte('date', $toDate)
            ->group_by('type')
            ->group_by_expr($groupExpr)
            ->order_by_expr($groupExpr);

        if (!$all_data) {
            $query->where('aid', $user->id);
        }

        $rows = $query->find_array();

        $income = [];
        $expense = [];

        foreach ($rows as $row) {
            $period = $row['period'];
            if ($row['type'] === 'Income') {
                $income[$period] = round((float)$row['total']);
            } elseif ($row['type'] === 'Expense') {
                $expense[$period] = round((float)$row['total']);
            }
        }

        // Normalizar periodos faltantes
        $labels = [];
        if ($groupDaily) {
            $cursor = $start->copy();
            while ($cursor->lte($end)) {
                $labels[] = $cursor->toDateString();
                $cursor->addDay();
            }
        } else {
            $cursor = $start->copy()->startOfMonth();
            $endMonth = $end->copy()->startOfMonth();
            while ($cursor->lte($endMonth)) {
                $labels[] = $cursor->format('Y-m-01');
                $cursor->addMonth();
            }
        }

        $incomeSeries = [];
        $expenseSeries = [];
        foreach ($labels as $label) {
            $incomeSeries[] = $income[$label] ?? 0;
            $expenseSeries[] = $expense[$label] ?? 0;
        }

        return [
            'Months' => $labels,
            'Income' => $incomeSeries,
            'Expense' => $expenseSeries,
        ];
    }

    private function getNetWorthRange($fromDate, $toDate, $allDataAccess, $user)
    {
        $income = ORM::for_table('sys_transactions')
            ->where('type', 'Income')
            ->where_gte('date', $fromDate)
            ->where_lte('date', $toDate);
        $expense = ORM::for_table('sys_transactions')
            ->where('type', 'Expense')
            ->where_gte('date', $fromDate)
            ->where_lte('date', $toDate);

        if (!$allDataAccess && isset($user->id)) {
            $income->where('aid', $user->id);
            $expense->where('aid', $user->id);
        }

        $incomeTotal = (float)$income->sum('amount');
        $expenseTotal = (float)$expense->sum('amount');

        return $incomeTotal - $expenseTotal;
    }

    private function getRecentSales($fromDate = null, $toDate = null)
    {
        $query = ORM::for_table('sys_transactions')
            ->table_alias('st')
            ->left_outer_join('sys_users', array('st.staff_id', '=', 'su.id'), 'su')
            ->select('st.*')
            ->select('su.fullname', 'staff_name')
            ->where('st.type', 'Income');

        if ($fromDate && $toDate) {
            $query = $query
                ->where_gte('st.date', $fromDate)
                ->where_lte('st.date', $toDate);
        }

        $rows = $query->order_by_desc('st.id')
            ->limit(6)
            ->find_array();

        $out = [];
        foreach ($rows as $row) {
            $title = $row['description'] ?? '';
            if ($title === '') {
                $title = $row['category'] ?? 'Income';
            }
            $dateRaw = $row['updated_at'] ?? $row['date'] ?? '';
            $timestampMs = null;
            try {
                if (!empty($dateRaw)) {
                    $tz = 'America/New_York'; // Virginia timezone
                    $timestampMs = Carbon::parse($dateRaw, $tz)->getTimestampMs();
                }
            } catch (\Throwable $e) {
                $timestampMs = null;
            }
            $out[] = [
                'title' => $title,
                'staff_name' => $row['staff_name'] ?? '',
                'date' => $dateRaw,
                'timestamp_ms' => $timestampMs,
                'amount' => $row['amount'] ?? 0,
            ];
        }

        return $out;
    }

    private function getStaffSalesRange($fromDateTime, $toDateTime, $allDataAccess, $user)
    {
        // Seleccionar de la tabla transacciones y sumar el monto agrupado por usuario
        $sql = "
            SELECT 
                st.staff_id_pago AS staff_id,
                su.fullname AS fullname,
                SUM(st.amount) AS total_amount
            FROM sys_transactions st
            LEFT JOIN sys_users su ON su.id = st.staff_id_pago
            WHERE (st.type = 'Income' OR st.type IN ('In','Out'))
                AND st.date BETWEEN :fromDate AND :toDate
        ";

        $params = [
            ':fromDate' => $fromDateTime,
            ':toDate' => $toDateTime,
        ];

        $sql .= " GROUP BY st.staff_id_pago, su.fullname ORDER BY total_amount DESC ";

        $pdo = ORM::get_db();
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$rows) {
            return ['labels' => [], 'values' => [], 'total' => 0, 'rows' => []];
        }

        $labels = [];
        $values = [];
        $total = 0;

        foreach ($rows as $row) {
            $labels[] = $row['fullname'];
            $amount = (float)($row['total_amount'] ?? 0);
            $values[] = $amount;
            $total += $amount;
        }


        return [
            'labels' => $labels,
            'values' => $values,
            'total' => $total,
            'rows' => $rows,
        ];
    }

    public function computeTopServicesChartData($aid = null, $limit = 5, $fromDate = null, $toDate = null)
    {
        // Obtener IDs de facturas dentro del rango solicitado
        $invoiceQuery = Invoice::query();
        if (!empty($aid)) {
            $invoiceQuery->where('aid', $aid);
        }
        if ($fromDate && $toDate) {
            $invoiceQuery->whereBetween('date', [$fromDate, $toDate]);
        }
        $invoiceIds = $invoiceQuery->select('id')->get()->pluck('id')->toArray();

        if (empty($invoiceIds)) {
            return ['labels' => [], 'values' => []];
        }

        // Obtener items de facturas filtradas
        if (!empty($aid)) {
            $invoiceItems = InvoiceItem::whereIn('invoiceid', $invoiceIds)
                ->select('itemcode', 'qty', 'amount', 'total')
                ->get();
        } else {
            $invoiceItems = InvoiceItem::whereIn('invoiceid', $invoiceIds)
                ->select('itemcode', 'qty', 'amount', 'total')
                ->get();
        }

        // Mapas de nombres (por item_number y por id) para resolver codigos numericos
        $items = Item::select('id', 'item_number', 'name')->get();
        $itemsByNumber = $items->keyBy('item_number');
        $itemsById = $items->keyBy('id');

        // Agregar por itemcode
        $agg = [];
        foreach ($invoiceItems as $row) {
            $code = $row->itemcode ?: '';
            if ($code === '') { // saltar registros sin codigo asignado
                continue;
            }

            $qty = is_numeric($row->qty) ? (float)$row->qty : (float)str_replace(',', '.', $row->qty);
            $amountTotal = isset($row->total) ? (float)$row->total : ((float)$row->amount * $qty);

            if (!isset($agg[$code])) {
                // Resolver etiqueta: intenta por item_number, luego por id, luego description, luego el code
                $label = $code;
                if (isset($itemsByNumber[$code])) {
                    $label = $itemsByNumber[$code]->name ?? $label;
                } elseif (isset($itemsById[$code])) {
                    $label = $itemsById[$code]->name ?? $label;
                } elseif (!empty($row->description)) {
                    $label = $row->description;
                }

                $agg[$code] = ['label' => $label, 'units' => 0.0, 'amount' => 0.0];
            }

            $agg[$code]['units'] += $qty;
            $agg[$code]['amount'] += $amountTotal;
        }
        if (empty($agg)) {
            return ['labels' => [], 'values' => []];
        }

        // Ordenar por unidades vendidas desc
        usort($agg, function ($a, $b) {
            if ($a['units'] === $b['units']) { return 0; }
            return ($a['units'] > $b['units']) ? -1 : 1;
        });

        // Top N sin agrupar "Otros"
        $top = array_slice($agg, 0, $limit);

        // Salida para bar chart: etiquetas y valores (unidades)
        $labels = array_map(function ($t) { return $t['label']; }, $top);
        $values = array_map(function ($t) { return (float)$t['units']; }, $top);

        return ['labels' => $labels, 'values' => $values];
    }

    private function percentageChange($current, $previous)
    {
        $current = (float)$current;
        $previous = (float)$previous;
        if ($previous == 0.0) {
            return 0;
        }
        return round((($current - $previous) / $previous) * 100, 1);
    }

    
}




