<?php
namespace App\Controllers;

//use Aws\SecretsManager\SecretsManagerClient;
use App\Controllers\BaseController;
use App\Models\Contact;
use ORM;

class PlaidController extends BaseController
{

    private $PLAID_CLIENT_ID;
    private $PLAID_SECRET;
    private $PLAID_ENV;

    public function __construct()
    {
        $this->PLAID_CLIENT_ID = $_ENV['PLAID_CLIENT_ID'];
        $this->PLAID_SECRET    = $_ENV['PLAID_SECRET'];
        $this->PLAID_ENV       = $_ENV['PLAID_ENV'];
    }

    private function plaidConfigured(): bool
    {
        return !empty($this->PLAID_CLIENT_ID) && !empty($this->PLAID_SECRET) && !empty($this->PLAID_ENV);
    }

    private function failNotConfigured(): bool
    {
        if ($this->plaidConfigured()) {
            return true;
        }
        header('Content-Type: application/json');
        http_response_code(500);
        echo json_encode(['error' => 'Plaid environment variables are missing or not set.']);
        return false;
    }

    private function plaidUrl($endpoint)
    {
        return "https://{$this->PLAID_ENV}.plaid.com/{$endpoint}";
    }

    private function plaidRequest($endpoint, $body)
    {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->plaidUrl($endpoint),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_POSTFIELDS => json_encode($body),
            CURLOPT_TIMEOUT => 30,
        ]);

        $resp = curl_exec($ch);
        if (curl_errno($ch)) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new \Exception('Plaid request error: ' . $err);
        }
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $decoded = json_decode($resp, true);
        if ($decoded === null) {
            throw new \Exception('Plaid response decode error or empty response (HTTP ' . $http_code . '): ' . $resp);
        }

        return $decoded;
    }

    private function encryptAccessToken($token)
    {
        $key = $_ENV['PLAID_ENCRYPTION_KEY'] ?? $_ENV['PLAID_SECRET'] ?? null;
        if (!$key) {
            throw new \Exception('Encryption key for Plaid tokens is not configured.');
        }

        $encryptionKey = hash('sha256', $key, true);
        $iv = random_bytes(16);
        $cipher = openssl_encrypt($token, 'aes-256-gcm', $encryptionKey, OPENSSL_RAW_DATA, $iv, $tag);

        if ($cipher === false) {
            throw new \Exception('Failed to encrypt Plaid access token.');
        }

        return base64_encode($iv . $tag . $cipher);
    }

    private function decryptAccessToken($encoded)
    {
        $key = $_ENV['PLAID_ENCRYPTION_KEY'] ?? $_ENV['PLAID_SECRET'] ?? null;
        if (!$key) {
            throw new \Exception('Encryption key for Plaid tokens is not configured.');
        }

        $decoded = base64_decode($encoded, true);
        if ($decoded === false || strlen($decoded) < 32) {
            throw new \Exception('Stored Plaid token is invalid.');
        }

        $iv = substr($decoded, 0, 16);
        $tag = substr($decoded, 16, 16);
        $cipher = substr($decoded, 32);

        $encryptionKey = hash('sha256', $key, true);
        $plain = openssl_decrypt($cipher, 'aes-256-gcm', $encryptionKey, OPENSSL_RAW_DATA, $iv, $tag);
        if ($plain === false) {
            throw new \Exception('Failed to decrypt Plaid access token.');
        }

        return $plain;
    }

    private function resolveClientId(array $payload = []): ?int
    {
        if (isset($payload['client_id'])) {
            $clean = preg_replace('/\D+/', '', (string) $payload['client_id']);
            if ($clean !== '') {
                return (int) $clean;
            }
        }

        $client = Contact::_detail();
        return isset($client->id) ? (int) $client->id : null;
    }

    private function getStoredPlaidConnection($clientId)
    {
        return ORM::for_table('plaid_connections')
            ->where('client_id', $clientId)
            ->where('status', 'active')
            ->order_by_desc('last_sync')
            ->order_by_desc('id')
            ->find_one();
    }

    private function storePlaidConnection($clientId, $itemId, $accessToken, $institutionName = null)
    {
        
        $clientId = (int) $clientId;

        $pdo = ORM::get_db();
        $finalStoredValue = $this->encryptAccessToken($accessToken);

        $sql = "INSERT INTO plaid_connections 
                (client_id, item_id, access_token, institution_name, status, last_sync) 
                VALUES 
                (:client_id, :item_id, :access_token, :institution_name, 'active', NOW())
                ON DUPLICATE KEY UPDATE 
                access_token = :access_token,
                status = 'active',
                last_sync = NOW()";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            'client_id'        => $clientId,
            'item_id'          => $itemId,
            'access_token'     => $finalStoredValue,
            'institution_name' => $institutionName
        ]);
    }

    // -------------------------
    // 1) Crear Link Token
    // POST /api/plaid/create_link_token
    // -------------------------
    public function createLinkToken()
    {
        header('Content-Type: application/json');
        if (!$this->failNotConfigured()) {
            return;
        }

        $clientId = $this->resolveClientId(['client_id' => $_GET['client_id'] ?? null]);

        if (!$clientId) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            return;
        }

        $body = [
            'client_id' => $this->PLAID_CLIENT_ID,
            'secret'    => $this->PLAID_SECRET,
            'user' => [
                'client_user_id' => (string)$clientId,
            ],
            'client_name'   => 'XHDE - Plaid',
            'products'      => ['transactions'],
            'language'      => 'en',
            'country_codes' => ['US']
        ];

        try {
            $res = $this->plaidRequest('link/token/create', $body);
            http_response_code(200);
            echo json_encode($res);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // -------------------------
    // 2) Intercambiar public_token -> access_token
    // POST /api/plaid/exchange_public_token
    // -------------------------
    public function exchangePublicToken()
    {
        header('Content-Type: application/json');
        if (!$this->failNotConfigured()) {
            return;
        }
        $data = json_decode(file_get_contents('php://input'), true);
        $clientId = $this->resolveClientId($data ?? []);

        if (empty($data['public_token'])) {
            http_response_code(400);
            echo json_encode(['error' => 'public_token is required']);
            return;
        }

        if (!$clientId) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            return;
        }

        try {
            $res = $this->plaidRequest('item/public_token/exchange', [
                'client_id' => $this->PLAID_CLIENT_ID,
                'secret'    => $this->PLAID_SECRET,
                'public_token' => $data['public_token'],
            ]);

            $access_token = $res['access_token'] ?? null;
            $item_id = $res['item_id'] ?? null;
            
            if ($access_token && $item_id) {
                $institutionName = null;

                // Fetch institution_id from item/get, then resolve name via institutions/get_by_id
                try {
                    $itemData = $this->plaidRequest('item/get', [
                        'client_id' => $this->PLAID_CLIENT_ID,
                        'secret'    => $this->PLAID_SECRET,
                        'access_token' => $access_token,
                    ]);
                    $institutionId = $itemData['item']['institution_id'] ?? null;

                    if ($institutionId) {
                        try {
                            $inst = $this->plaidRequest('institutions/get_by_id', [
                                'client_id'      => $this->PLAID_CLIENT_ID,
                                'secret'         => $this->PLAID_SECRET,
                                'institution_id' => $institutionId,
                                'country_codes'  => ['US'],
                            ]);
                            $institutionName = $inst['institution']['name'] ?? null;
                        } catch (\Exception $instErr) {
                            error_log('Plaid institution lookup failed: ' . $instErr->getMessage());
                        }
                    }
                } catch (\Exception $itemErr) {
                    error_log('Plaid item lookup failed: ' . $itemErr->getMessage());
                }

                try {
                    $this->storePlaidConnection($clientId, $item_id, $access_token, $institutionName);
                } catch (\Exception $storageError) {
                    error_log('Plaid token storage failed: ' . $storageError->getMessage());
                    http_response_code(500);
                    echo json_encode(['error' => 'Error trying to save connection.']);
                    return;
                }
            }

            http_response_code(200);
            echo json_encode($res);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // -------------------------
    // 3) Obtener transacciones por access_token
    // POST /api/plaid/transactions
    // Body JSON: { "access_token":"...", "start_date":"YYYY-MM-DD", "end_date":"YYYY-MM-DD", "count":100, "offset":0 }
    // -------------------------
    public function getTransactions()
    {
        header('Content-Type: application/json');
        if (!$this->failNotConfigured()) {
            return;
        }
        $data = json_decode(file_get_contents('php://input'), true) ?: [];

        $clientId = $this->resolveClientId($data);

        $start_date = $data['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
        $end_date   = $data['end_date'] ?? date('Y-m-d');
        $count      = isset($data['count']) ? (int)$data['count'] : 100;
        $offset     = isset($data['offset']) ? (int)$data['offset'] : 0;

        if (!$clientId) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            return;
        }

        $connection = $this->getStoredPlaidConnection($clientId);
        if (!$connection) {
            http_response_code(404);
            echo json_encode(['error' => 'No active Plaid connection found']);
            return;
        }

        try {
            $access_token = $this->decryptAccessToken($connection->access_token);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Plaid access token could not be decrypted']);
            return;
        }

        $body = [
            'client_id' => $this->PLAID_CLIENT_ID,
            'secret'    => $this->PLAID_SECRET,
            'access_token' => $access_token,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'options' => [
                'count' => $count,
                'offset' => $offset
            ]
        ];

        try {
            $res = $this->plaidRequest('transactions/get', $body);
            http_response_code(200);
            echo json_encode($res);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // -------------------------
    // 4) Estado de conexión
    // GET /api/plaid/connection/status
    // -------------------------
    public function connectionStatus()
    {
        header('Content-Type: application/json');
        if (!$this->failNotConfigured()) {
            return;
        }
        $clientId = $this->resolveClientId(['client_id' => $_GET['client_id'] ?? null]);

        if (!$clientId) {
            http_response_code(400);
            echo json_encode(['connected' => false, 'error' => 'Client not found for user']);
            return;
        }

        $connection = $this->getStoredPlaidConnection($clientId);
        if (!$connection) {
            http_response_code(200);
            echo json_encode(['connected' => false]);
            return;
        }

        http_response_code(200);
        echo json_encode([
            'connected' => true,
            'item_id' => $connection->item_id,
            'institution_name' => $connection->institution_name,
            'status' => $connection->status,
            'last_sync' => $connection->last_sync
        ]);
    }


    /*private function getSecretsFromAWS() {
        $client = new SecretsManagerClient([
            'version' => 'latest',
            'region'  => $_ENV['AWS_REGION'],
            'credentials' => [
                'key'    => $_ENV['AWS_ACCESS_KEY_ID'],
                'secret' => $_ENV['AWS_SECRET_ACCESS_KEY'],
            ],
        ]);

        $result = $client->getSecretValue(['SecretId' => $_ENV['AWS_PLAID_SECRET_NAME']]);
        return json_decode($result['SecretString'], true);
    }*/

}
