(function (global) {
    class SmartyAddress {
        constructor(options = {}) {
            const baseUrl = (global.BASE_URL || global.APP_URL || '').replace(/\/$/, '');
            const defaults = {
                countryValue: 'United States',
                minChars: 2,
                debounceMs: 300,
                suggestUrl: `${baseUrl}/api/address/suggest`,
                selectUnitUrl: `${baseUrl}/api/address/select-unit`,
                suggestionClass: 'smarty-address-suggestions',
                fetchOptions: { credentials: 'same-origin' },
                clearFieldError: null, // optional callback(fieldName, context)
                onError: null, // optional callback(message, error, context)
                onSelect: null, // optional callback({ context, suggestion })
                address: {
                    country: '#country',
                    input: "input[name='address']",
                    city: '#city',
                    state: '#state',
                    zip: '#zip',
                    suggestions: null,
                },
                billing: null,
            };

            this.opts = this.mergeDeep(defaults, options);
            this.addressTimeout = null;
            this.billingTimeout = null;
            this.addressSuggestions = [];
            this.billAddressSuggestions = [];
            this.unitLoading = null;
            this.previousSuggestions = {
                address: [],
                billing: [],
            };

            this.dom = {
                address: this.buildDomHandles(this.opts.address),
                billing: this.opts.billing ? this.buildDomHandles(this.opts.billing) : null,
            };

            this.bindHandlers();
            this.attachListeners();
        }

        mergeDeep(target, source) {
            const output = { ...target };
            Object.keys(source || {}).forEach((key) => {
                if (source[key] && typeof source[key] === 'object' && !Array.isArray(source[key])) {
                    output[key] = this.mergeDeep(target[key] || {}, source[key]);
                } else {
                    output[key] = source[key];
                }
            });
            return output;
        }

        queryElement(ref) {
            if (!ref) return null;
            if (typeof ref === 'string') return document.querySelector(ref);
            return ref instanceof Element ? ref : null;
        }

        buildDomHandles(config = {}) {
            const input = this.queryElement(config.input);
            const container = this.queryElement(config.suggestions) || this.createSuggestionContainer(input);
            if (container) {
                container.classList.add(this.opts.suggestionClass, 'autocomplete-suggestions');
            }
            return {
                country: this.queryElement(config.country),
                input,
                city: this.queryElement(config.city),
                state: this.queryElement(config.state),
                zip: this.queryElement(config.zip),
                suggestions: container,
            };
        }

        createSuggestionContainer(input) {
            const box = document.createElement('div');
            box.className = `${this.opts.suggestionClass} autocomplete-suggestions`;
            box.style.position = 'absolute';
            box.style.zIndex = '9999';
            box.style.width = '100%';
            box.style.display = 'none';
            if (input && input.parentNode) {
                const wrapper = input.parentNode;
                const computed = window.getComputedStyle(wrapper);
                if (computed.position === 'static') {
                    wrapper.style.position = 'relative';
                }
                wrapper.appendChild(box);
            }
            return box;
        }

        bindHandlers() {
            this.handleAddressInput = this.handleAddressInput.bind(this);
            this.handleBillingAddressInput = this.handleBillingAddressInput.bind(this);
            this.handleAddressFocus = this.handleAddressFocus.bind(this);
            this.handleBillingFocus = this.handleBillingFocus.bind(this);
            this.handleOutsideClick = this.handleOutsideClick.bind(this);
        }

        attachListeners() {
            if (this.dom.address.input) {
                this.dom.address.input.addEventListener('input', this.handleAddressInput);
                this.dom.address.input.addEventListener('focus', this.handleAddressFocus);
            }

            if (this.dom.billing?.input) {
                this.dom.billing.input.addEventListener('input', this.handleBillingAddressInput);
                this.dom.billing.input.addEventListener('focus', this.handleBillingFocus);
            }

            document.addEventListener('click', this.handleOutsideClick);
        }

        detachListeners() {
            if (this.dom.address.input) {
                this.dom.address.input.removeEventListener('input', this.handleAddressInput);
                this.dom.address.input.removeEventListener('focus', this.handleAddressFocus);
            }
            if (this.dom.billing?.input) {
                this.dom.billing.input.removeEventListener('input', this.handleBillingAddressInput);
                this.dom.billing.input.removeEventListener('focus', this.handleBillingFocus);
            }
            document.removeEventListener('click', this.handleOutsideClick);
        }

        destroy() {
            this.detachListeners();
            this.clearSuggestions('address');
            this.clearSuggestions('billing');
        }

        handleOutsideClick(event) {
            const containers = [
                this.dom.address?.suggestions,
                this.dom.billing?.suggestions,
            ].filter(Boolean);

            const clickedInside = containers.some((container) => container.contains(event.target));
            const clickedInput = [this.dom.address?.input, this.dom.billing?.input].includes(event.target);

            if (!clickedInside && !clickedInput) {
                this.clearSuggestions('address');
                this.clearSuggestions('billing');
            }
        }

        isUnitedStates(context) {
            const field = this.dom[context]?.country;
            const value = field ? (field.value || field.textContent || '').trim() : '';
            return value.toLowerCase() === this.opts.countryValue.toLowerCase();
        }

        handleAddressInput(event) {
            this.processInput(event, 'address');
        }

        handleBillingAddressInput(event) {
            this.processInput(event, 'billing');
        }

        handleAddressFocus() {
            this.processFocus('address');
        }

        handleBillingFocus() {
            this.processFocus('billing');
        }

        processFocus(context) {
            const dom = this.dom[context];
            if (!dom?.input) return;
            if (!this.isUnitedStates(context)) return;

            const current = dom.input.value || '';
            if (current.trim().length >= this.opts.minChars) {
                this.searchAddress(current, context);
            }
        }

        processInput(event, context) {
            if (!this.dom[context]?.input) return;

            if (!this.isUnitedStates(context)) {
                this.clearSuggestions(context);
                return;
            }

            const query = (event.target.value || '').trim();
            if (query.length < this.opts.minChars) {
                this.clearSuggestions(context);
                return;
            }

            const timeoutKey = context === 'billing' ? 'billingTimeout' : 'addressTimeout';
            clearTimeout(this[timeoutKey]);
            this[timeoutKey] = setTimeout(() => {
                this.searchAddress(query, context);
            }, this.opts.debounceMs);
        }

        async searchAddress(query, context) {
            try {
                const url = new URL(this.opts.suggestUrl, window.location.origin);
                url.searchParams.set('query', query);
                const response = await this.fetchJson(url.toString());
                const suggestions = response.suggestions || [];
                this.setSuggestions(context, suggestions);
            } catch (error) {
                if (String(error).includes('Missing Smarty credentials')) {
                    this.clearSuggestions(context);
                    return;
                }
                this.handleError('Error fetching address suggestions', error, context);
            }
        }

        async handleUnitClick(suggestion, context, event) {
            if (event) event.stopPropagation();
            try {
                this.unitLoading = suggestion;
                if (context === 'billing') {
                    this.previousSuggestions.billing = [...this.billAddressSuggestions];
                } else {
                    this.previousSuggestions.address = [...this.addressSuggestions];
                }

                const baseSelected = this.formatBaseSelectedValue(suggestion);
                const unitSelected = this.formatUnitSelectedValue(suggestion) || baseSelected;
                const params = new URLSearchParams({
                    search: suggestion.value?.street_line || baseSelected || suggestion.label || this.formatSuggestionLabel(suggestion),
                    selected: unitSelected || '',
                });

                const url = `${this.opts.selectUnitUrl}?${params.toString()}`;
                const response = await this.fetchJson(url);
                const suggestions = response.suggestions || [];
                const suggestionsWithBack = [
                    { isBack: true, label: '← Back', value: {}, className: 'autocomplete-back' },
                    ...suggestions,
                ];
                this.setSuggestions(context, suggestionsWithBack);

            } catch (error) {
                if (String(error).includes('Missing Smarty credentials')) {
                    this.clearSuggestions(context);
                    return;
                }
                this.handleError('Error fetching units suggestions', error, context);
            } finally {
                this.unitLoading = null;
            }
        }

        setSuggestions(context, suggestions) {
            if (context === 'billing') {
                this.billAddressSuggestions = suggestions;
            } else {
                this.addressSuggestions = suggestions;
            }
            this.renderSuggestions(context, suggestions);
        }

        clearSuggestions(context) {
            const dom = this.dom[context];
            if (!dom?.suggestions) return;
            dom.suggestions.innerHTML = '';
            dom.suggestions.style.display = 'none';
            if (context === 'billing') {
                this.billAddressSuggestions = [];
            } else {
                this.addressSuggestions = [];
            }
        }

        renderSuggestions(context, suggestions) {
            const dom = this.dom[context];
            if (!dom?.suggestions) return;

            dom.suggestions.innerHTML = '';
            if (!suggestions || !suggestions.length) {
                dom.suggestions.style.display = 'none';
                return;
            }

            const fragment = document.createDocumentFragment();

            suggestions.forEach((suggestion) => {
                const item = document.createElement('div');
                item.className = `${this.opts.suggestionClass}-item`;

                if (suggestion.isBack) {
                    const backButton = document.createElement('button');
                    backButton.type = 'button';
                    backButton.className = `${this.opts.suggestionClass}-option${suggestion.className ? ` ${suggestion.className}` : ''}`;
                    backButton.textContent = suggestion.label || '<- Back';
                    backButton.addEventListener('click', (event) => {
                        event.preventDefault();
                        event.stopPropagation();
                        const previous = context === 'billing'
                            ? this.previousSuggestions.billing
                            : this.previousSuggestions.address;
                        if (previous && previous.length) {
                            this.setSuggestions(context, previous);
                        } else {
                            this.clearSuggestions(context);
                        }
                    });
                    item.appendChild(backButton);
                    fragment.appendChild(item);
                    return;
                }

                const selectButton = document.createElement('button');
                selectButton.type = 'button';
                selectButton.className = `${this.opts.suggestionClass}-option`;
                selectButton.textContent = suggestion.value?.full_address
                    ? suggestion.value.full_address
                    : this.formatSuggestionLabel(suggestion);
                selectButton.addEventListener('click', (event) => {
                    event.preventDefault();
                    this.selectAddress(suggestion, context);
                });

                item.appendChild(selectButton);

                if (suggestion.value?.entries > 1) {
                    const unitButton = document.createElement('button');
                    unitButton.type = 'button';
                    unitButton.className = `${this.opts.suggestionClass}-unit`;
                    unitButton.textContent = this.unitLoading === suggestion
                        ? 'Loading units...'
                        : `See ${suggestion.value.entries} units`;
                    unitButton.disabled = this.unitLoading === suggestion;
                    unitButton.addEventListener('click', (event) => this.handleUnitClick(suggestion, context, event));
                    item.appendChild(unitButton);
                }

                fragment.appendChild(item);
            });

            dom.suggestions.appendChild(fragment);
            dom.suggestions.style.display = 'block';
        }

        formatSuggestionLabel(suggestion) {
            if (suggestion.label) return suggestion.label;
            const value = suggestion.value || {};
            const parts = [
                value.street_line,
                value.secondary,
                [value.city, value.state].filter(Boolean).join(', '),
                value.zipcode,
            ].filter(Boolean);
            return parts.join(' ');
        }

        formatSelectedValue(suggestion) {
            const value = suggestion.value || {};
            return [
                suggestion.label || '',
                value.secondary || '',
                value.entries ? `(${value.entries})` : '',
                value.city || '',
                value.state || '',
                value.zipcode || '',
            ]
                .filter(Boolean)
                .join(' ')
                .trim();
        }

        formatBaseSelectedValue(suggestion) {
            const value = suggestion.value || {};
            return [
                value.street_line || '',
                value.city || '',
                value.state || '',
                value.zipcode || '',
            ]
                .filter(Boolean)
                .join(' ')
                .trim();
        }

        formatUnitSelectedValue(suggestion) {
            const value = suggestion.value || {};
            const entries = value.entries || 1;
            return [
                value.street_line || '',
                value.secondary || '',
                `(${entries})`,
                value.city || '',
                value.state || '',
                value.zipcode || '',
            ]
                .filter(Boolean)
                .join(' ')
                .trim();
        }

        selectAddress(suggestion, context) {
            const dom = this.dom[context];
            const value = suggestion.value || {};
            if (dom?.input) {
                dom.input.value = value.street_line + (value.secondary ? ` ${value.secondary}` : '');
            }
            if (dom?.city) dom.city.value = value.city || '';
            if (dom?.state) dom.state.value = value.state || '';
            if (dom?.zip) dom.zip.value = value.zipcode || '';

            if (typeof this.opts.clearFieldError === 'function') {
                this.opts.clearFieldError('address', context);
            }

            if (typeof this.opts.onSelect === 'function') {
                this.opts.onSelect({ context, suggestion });
            }

            this.clearSuggestions(context);
        }

        async fetchJson(url) {
            const response = await fetch(url, this.opts.fetchOptions);
            const text = await response.text();
            if (!response.ok) {
                throw new Error(text || `Request failed with status ${response.status}`);
            }
            if (!text) {
                return {};
            }
            try {
                return JSON.parse(text);
            } catch (parseError) {
                throw new Error(`Invalid JSON response: ${parseError.message} | Body: ${text}`);
            }
        }

        handleError(message, error, context) {
            console.error(message, error);
            if (typeof this.opts.onError === 'function') {
                this.opts.onError(message, error, context);
            }
        }
    }

    global.SmartyAddress = SmartyAddress;
}(window));


