<?php
namespace App\Controllers;
use ORM;
use PDO;
use App\Models\User as User;
use App\Models\Invoice as Invoice;
use App\Models\InvoiceItem as InvoiceItem;
use App\Models\Item as Item;
use App\Models\Project as Project;
use App\Models\Contact as Contact;
use App\Models\Transaction as Transaction;
use Carbon\Carbon;

require_once __DIR__ . '/BaseController.php';
require_once __DIR__ . '/../Models/User.php';
require_once __DIR__ . '/../Models/Invoice.php';
require_once __DIR__ . '/../Models/InvoiceItem.php';
require_once __DIR__ . '/../Models/Item.php';
require_once __DIR__ . '/../Models/Project.php';
require_once __DIR__ . '/../Models/Contact.php';
require_once __DIR__ . '/../Models/Transaction.php';
require_once __DIR__ . '/../util/global.php';

class EmployeeDashboardController extends BaseController
{
    public function getDashboardData($all_data_access = false)
    {
        header('Content-Type: application/json');

        $user = User::_info();
        $GLOBALS['user'] = $user;

        global $config;

        $payloadRaw = file_get_contents('php://input');
        $payload = json_decode($payloadRaw, true);
        if (!is_array($payload)) {
            $payload = $_REQUEST;
        }
        if (!is_array($payload)) {
            http_response_code(400);
            echo json_encode([
                'type' => 'error',
                'message' => 'Invalid payload',
            ]);
            exit;
        }


        $rangeParam = _get('range') ?? 'weekly';
        $customStart = _get('start_date') ?? null;
        $customEnd = _get('end_date') ?? null;
        $startDate = $payload['startDate'] ?? null;
        $endDate = $payload['endDate'] ?? null;

        $now = Carbon::now();
        switch ($rangeParam) {
            case 'monthly':
                $rangeStart = $now->copy()->startOfMonth();
                $rangeEnd = $now->copy()->endOfMonth();
                break;
            case 'yearly':
                $rangeStart = $now->copy()->startOfYear();
                $rangeEnd = $now->copy()->endOfYear();
                break;
            case 'custom':
                try {
                    $rangeStart = Carbon::parse($customStart);
                    $rangeEnd = $customEnd ? Carbon::parse($customEnd) : Carbon::parse($customStart);
                    if ($rangeEnd->lessThan($rangeStart)) {
                        [$rangeStart, $rangeEnd] = [$rangeEnd, $rangeStart];
                    }
                } catch (\Throwable $e) {
                    $rangeStart = $now->copy()->startOfWeek(Carbon::MONDAY);
                    $rangeEnd = $now->copy()->startOfWeek(Carbon::MONDAY)->addDays(5);
                }
                break;
            case 'weekly':
            default:
                $rangeStart = $now->copy()->startOfWeek(Carbon::MONDAY);
                $rangeEnd = $now->copy()->startOfWeek(Carbon::MONDAY)->addDays(5);
                break;
        }

        $rangeDays = $rangeStart->diffInDays($rangeEnd) + 1;
        $prevRangeEnd = $rangeStart->copy()->subDay();
        $prevRangeStart = $prevRangeEnd->copy()->subDays($rangeDays - 1);

        $rangeStartDate = $rangeStart->toDateString();
        $rangeEndDate = $rangeEnd->toDateString();
        $prevRangeStartDate = $prevRangeStart->toDateString();
        $prevRangeEndDate = $prevRangeEnd->toDateString();

        if (empty($startDate) || empty($endDate)) {
            $startDate = $rangeStartDate;
            $endDate = $rangeEndDate;
        }

        if (empty($startDate) || empty($endDate)) {
            http_response_code(422);
            echo json_encode([
                'type' => 'error',
                'message' => 'Incorrect date!',
            ]);
            exit;
        }


        $invoiceQuery = Invoice::whereBetween('date', [$startDate, $endDate]);

        if (!$all_data_access && isset($user->id)) {
            $invoiceQuery = $invoiceQuery->where('aid', $user->id);
        }

        $invoices = $invoiceQuery->get();

        $appointments = ORM::for_table('app_appointments')
            ->where('admin_id', $user->id)
            ->limit(5)
            ->order_by_desc('id')
            ->find_many();

        $total = 0;
        $currency = '';
        $invoices_id = [];
        $invoices_data = [];

        foreach ($invoices as $invoice) {
            $total += $invoice['credit'];
            $currency = $invoice['currency_iso_code'];

            $invoices_id[] = $invoice['id'];

            $invoiceUrl = U . 'invoices/view/' . $invoice['id'] . '/';
            $invoiceNumber = $invoice['invoicenum'];
            if ($invoice['cn'] != '') {
                $invoiceNumber .= ' ' . $invoice['cn'];
            } else {
                $invoiceNumber .= ' ' . $invoice['id'];
            }

            $invoices_data[] = [
                'id' => $invoice['id'],
                'invoice_url' => $invoiceUrl,
                'invoice_number' => $invoiceNumber,
                'account' => $invoice['account'],
                'credit' => $invoice['credit'],
                'currency_iso_code' => $invoice['currency_iso_code'],
                'credit_formatted' => formatCurrency($invoice['credit'], $invoice['currency_iso_code']),
                'title' => $invoice['title'],
                'date' => $invoice['date'],
                'date_formatted' => !empty($invoice['date']) ? date($config['df'], strtotime($invoice['date'])) : '--',
                'duedate' => $invoice['duedate'],
                'duedate_formatted' => !empty($invoice['duedate']) ? date($config['df'], strtotime($invoice['duedate'])) : '--',
                'status' => $invoice['status'],
                'status_label' => ib_lan_get_line($invoice['status']),
            ];
        }

        $item_sold = [];

        if (count($invoices) !== 0) {
            $all_invoice_items = InvoiceItem::where('itemcode', '!=', '')
                ->whereBetween('created_at', [$startDate . ' 00:00:01', $endDate . ' 23:59:59'])
                ->whereIn('invoiceid', $invoices_id)
                ->get()
                ->groupBy('itemcode')
                ->all();

            foreach ($all_invoice_items as $itemCode => $items) {
                $count = count($items);
                $amount = 0;
                $owner_name = '';

                foreach ($items as $item) {
                    $amount += $item->amount * $item->qty;
                    $owner = ORM::for_table('sys_items')->find($item->itemcode);
                    $owner_name = $owner ? $owner->name : '';
                }

                $item_sold[] = [
                    'name' => $owner_name,
                    'sold_count' => $count,
                    'total_amount' => $amount,
                ];
            }
        }

        $response = array(
            "type" => "success",
            "message" => "Information Saved Successfully!",
            "startDate" => $startDate,
            "endDate" => $endDate,
            "invoices" => $invoices_data,
            "total_invoices" => formatCurrency($total, $currency),
            "appointments" => $appointments,
            "item_sold_circular" => array_slice($item_sold, 0, 10),
        );

        http_response_code(200);
        echo json_encode($response);
        exit;
    }

}
