<?php
namespace App\Controllers;

require_once __DIR__ . '/BaseController.php';
require_once __DIR__ . '/../Models/User.php';

use App\Models\User;

class PlaidController extends BaseController
{

    private $PLAID_CLIENT_ID;
    private $PLAID_SECRET;
    private $PLAID_ENV;

    public function __construct()
    {
        $this->PLAID_CLIENT_ID = "XXXXXXXXXXXXXXXXX";
        $this->PLAID_SECRET    = "XXXXXXXXXXXXXXXXX";
        $this->PLAID_ENV       = "sandbox";
    }

    private function plaidUrl($endpoint)
    {
        return "https://{$this->PLAID_ENV}.plaid.com/{$endpoint}";
    }

    private function plaidRequest($endpoint, $body)
    {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->plaidUrl($endpoint),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_POSTFIELDS => json_encode($body),
            CURLOPT_TIMEOUT => 30,
        ]);

        $resp = curl_exec($ch);
        if (curl_errno($ch)) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new \Exception('Plaid request error: ' . $err);
        }
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $decoded = json_decode($resp, true);
        if ($decoded === null) {
            throw new \Exception('Plaid response decode error or empty response (HTTP ' . $http_code . '): ' . $resp);
        }

        return $decoded;
    }

    // -------------------------
    // 1) Crear Link Token
    // POST /api/plaid/create_link_token
    // -------------------------
    public function createLinkToken()
    {
        header('Content-Type: application/json');

        // Get auth user
        $user = User::_info();
        if (!$user) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            return;
        }

        $client_user_id = (string)$user->id;

        $body = [
            'client_id' => $this->PLAID_CLIENT_ID,
            'secret'    => $this->PLAID_SECRET,
            'user' => [
                'client_user_id' => $client_user_id,
            ],
            'client_name'   => 'XHDE - Plaid',
            'products'      => ['transactions'],
            'language'      => 'en',
            'country_codes' => ['US']
        ];

        try {
            $res = $this->plaidRequest('link/token/create', $body);
            http_response_code(200);
            echo json_encode($res);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // -------------------------
    // 2) Intercambiar public_token -> access_token
    // POST /api/plaid/exchange_public_token
    // -------------------------
    public function exchangePublicToken()
    {
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if (empty($data['public_token'])) {
            http_response_code(400);
            echo json_encode(['error' => 'public_token is required']);
            return;
        }

        try {
            $res = $this->plaidRequest('item/public_token/exchange', [
                'client_id' => $this->PLAID_CLIENT_ID,
                'secret'    => $this->PLAID_SECRET,
                'public_token' => $data['public_token'],
            ]);

            $user = User::_info();
            if ($user) {
                $access_token = $res['access_token'] ?? null;
                $item_id = $res['item_id'] ?? null;
                if ($access_token) {
                    //
                }
            }

            http_response_code(200);
            echo json_encode($res);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    // -------------------------
    // 3) Obtener transacciones por access_token
    // POST /api/plaid/transactions
    // Body JSON: { "access_token":"...", "start_date":"YYYY-MM-DD", "end_date":"YYYY-MM-DD", "count":100, "offset":0 }
    // -------------------------
    public function getTransactions()
    {
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        $access_token = $data['access_token'] ?? null;
        $start_date = $data['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
        $end_date   = $data['end_date'] ?? date('Y-m-d');
        $count      = isset($data['count']) ? (int)$data['count'] : 100;
        $offset     = isset($data['offset']) ? (int)$data['offset'] : 0;

        if (!$access_token) {
            http_response_code(400);
            echo json_encode(['error' => 'access_token is required']);
            return;
        }

        $body = [
            'client_id' => $this->PLAID_CLIENT_ID,
            'secret'    => $this->PLAID_SECRET,
            'access_token' => $access_token,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'options' => [
                'count' => $count,
                'offset' => $offset
            ]
        ];

        try {
            $res = $this->plaidRequest('transactions/get', $body);
            http_response_code(200);
            echo json_encode($res);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    /*public function getTransactionsV2()
    {
        header('Content-Type: application/json');
        $input = json_decode(file_get_contents('php://input'), true);
        $access_token = $input['access_token'];

        if (!$access_token) {
            echo json_encode(['error' => 'Falta access_token']);
            return;
        }

        $client = new Plaid\ApiClient([
            'clientId' => 'your_client_id',
            'secret' => 'your_sandbox_secret',
            'plaidEnv' => 'sandbox',
        ]);

        try {
            $response = $client->transactionsGet([
                'access_token' => $access_token,
                'start_date' => date('Y-m-d', strtotime('-30 days')),
                'end_date' => date('Y-m-d'),
            ]);

            echo json_encode($response->toArray());
        } catch (Exception $e) {
            echo json_encode(['error' => $e->getMessage()]);
        }
    }*/

}
