<?php
namespace App\Controllers;

use App\Controllers\BaseController;
use App\Library\UUIDHelper;
use ORM;
use Stripe\StripeClient;

class BillingController extends BaseController
{
    public function createCustomer()
    {
        $payload = $this->getJsonBody();
        $crmCustomerId = (int) ($payload['crm_customer_id'] ?? _post('crm_customer_id'));

        if ($crmCustomerId <= 0) {
            $this->jsonResponse(['error' => 'crm_customer_id is required'], 400);
        }

        $contact = ORM::for_table('crm_accounts')->find_one($crmCustomerId);
        if (!$contact) {
            $this->jsonResponse(['error' => 'Customer not found'], 404);
        }

        $stripeCustomerColumn = $this->resolveStripeCustomerColumn();
        if ($stripeCustomerColumn) {
            $existing = trim((string) ($contact->$stripeCustomerColumn ?? ''));
            if (!$this->isStripeCustomerId($existing)) {
                $existing = '';
            }
            if ($existing !== '') {
                try {
                    $stripe = $this->getStripeClient();
                    $stripe->customers->retrieve($existing, []);
                    $this->jsonResponse([
                        'stripe_customer_id' => $existing,
                        'crm_customer_id' => $crmCustomerId,
                    ]);
                } catch (\Throwable $e) {
                    // fall through to create a fresh customer
                }
            }
        }

        $email = trim((string) ($payload['email'] ?? $contact->email ?? ''));
        $name = trim((string) ($payload['name'] ?? $contact->account ?? ''));

        $params = [];
        if ($email !== '') {
            $params['email'] = $email;
        }
        if ($name !== '') {
            $params['name'] = $name;
        }

        try {
            $stripe = $stripe ?? $this->getStripeClient();
            $customer = null;

            try {
                $search = $stripe->customers->search([
                    'query' => "metadata['crm_customer_id']:'{$crmCustomerId}'",
                    'limit' => 1,
                ]);
                if (!empty($search->data)) {
                    $customer = $search->data[0];
                }
            } catch (\Throwable $e) {
                // search not available or failed
            }

            if (!$customer && $email !== '') {
                try {
                    $list = $stripe->customers->all([
                        'email' => $email,
                        'limit' => 1,
                    ]);
                    if (!empty($list->data)) {
                        $customer = $list->data[0];
                    }
                } catch (\Throwable $e) {
                    // ignore and create new
                }
            }

            if ($customer) {
                try {
                    $stripe->customers->update($customer->id, [
                        'metadata' => ['crm_customer_id' => (string) $crmCustomerId],
                    ]);
                } catch (\Throwable $e) {
                    // best effort metadata update
                }
            } else {
                $params['metadata'] = ['crm_customer_id' => (string) $crmCustomerId];
                $customer = $stripe->customers->create($params);
            }
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        if ($stripeCustomerColumn) {
            $contact->$stripeCustomerColumn = $customer->id;
            $contact->save();
        }

        $this->jsonResponse([
            'stripe_customer_id' => $customer->id,
            'crm_customer_id' => $crmCustomerId,
        ]);
    }

    public function getConfig()
    {
        $publishableKey = $this->getStripePublishableKey();

        $this->jsonResponse([
            'publishable_key' => $publishableKey,
        ]);
    }

    public function listPlans()
    {
        $plans = ORM::for_table('subscription_plans')
            ->where_not_equal('stripe_pricing_id', '')
            ->order_by_desc('id')
            ->find_many();

        $result = [];
        foreach ($plans as $plan) {
            $result[] = [
                'id' => $plan->id,
                'title' => $plan->title,
                'price' => $plan->price,
                'term' => $plan->term,
                'stripe_pricing_id' => $plan->stripe_pricing_id,
            ];
        }

        $this->jsonResponse(['data' => $result]);
    }

    public function listGateways()
    {
        $gateways = ORM::for_table('sys_pg')
            ->where('status', 'Active')
            ->order_by_desc('id')
            ->find_many();

        $result = [];
        foreach ($gateways as $gateway) {
            $result[] = [
                'id' => $gateway->id,
                'name' => $gateway->name ?? '',
                'identificador' => $gateway->identificador ?? '',
                'processor' => $gateway->processor ?? '',
            ];
        }

        $this->jsonResponse(['data' => $result]);
    }

    public function listSubscriptions()
    {
        $payload = $this->getJsonBody();
        $stripeCustomerId = trim((string) ($payload['stripe_customer_id'] ?? _post('stripe_customer_id')));
        $debugInvoiceId = trim((string) ($payload['debug_invoice_id'] ?? _post('debug_invoice_id')));
        $debugInvoiceSeen = false;

        if ($stripeCustomerId === '') {
            $this->jsonResponse(['error' => 'stripe_customer_id is required'], 400);
        }

        try {
            $stripe = $this->getStripeClient();
            $subs = $stripe->subscriptions->all([
                'customer' => $stripeCustomerId,
                'status' => 'all',
                'limit' => 100
            ]);
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        $invoiceIds = [];
        foreach ($subs->data as $sub) {
            $crmInvoiceId = $sub->metadata->crm_invoice_id ?? null;
            if ($crmInvoiceId) {
                $invoiceIds[] = (int) $crmInvoiceId;
            }
        }

        $scheduledInvoices = [];
        $oneTimeInvoices = [];
        try {
            $invoices = $stripe->invoices->all([
                'customer' => $stripeCustomerId,
                'limit' => 100
            ]);

            foreach ($invoices->data as $invoice) {

                $crmInvoiceId = $invoice->metadata->crm_invoice_id ?? null;
                if ($crmInvoiceId) {
                    $invoiceIds[] = (int)$crmInvoiceId;
                }
            
                $autoAdvance = !empty($invoice->auto_advance);
                $finalizesAt = (int) ($invoice->automatically_finalizes_at ?? 0);
                $isScheduled = $autoAdvance && $finalizesAt > time();
            
                if ($isScheduled && ($invoice->status ?? '') === 'draft') {
                    $scheduledInvoices[] = $invoice;
                    continue;
                }
            
                // if (!empty($invoice->subscription)) continue;
            
                if (in_array($invoice->status ?? '', ['paid', 'open', 'void', 'uncollectible'], true)) {
                    $oneTimeInvoices[] = $invoice;
                }
            }
            
        } catch (\Throwable $e) {
            $scheduledInvoices = [];
            $oneTimeInvoices = [];
        }

        $invoiceMap = [];
        if (!empty($invoiceIds)) {
            $rows = ORM::for_table('sys_invoices')
                ->select('id')
                ->select('invoicenum')
                ->select('status')
                ->where_in('id', array_values(array_unique($invoiceIds)))
                ->find_many();
            foreach ($rows as $row) {
                $invoiceMap[(int) $row->id] = [
                    'status' => $row->status,
                    'invoicenum' => $row->invoicenum,
                ];
            }
        }

        $result = [];
        foreach ($subs->data as $sub) {
            $price = $sub->items->data[0]->price ?? null;
            $crmInvoiceId = $sub->metadata->crm_invoice_id ?? null;
            $crmInvoiceStatus = null;
            $crmInvoiceNumber = null;
            if ($crmInvoiceId) {
                $map = $invoiceMap[(int) $crmInvoiceId] ?? null;
                $crmInvoiceStatus = $map['status'] ?? null;
                $crmInvoiceNumber = $map['invoicenum'] ?? null;
            }

            $result[] = [
                'id' => $sub->id,
                'status' => $sub->status,
                'price_id' => $price->id ?? null,
                'amount' => $price->unit_amount ?? null,
                'currency' => $price->currency ?? null,
                'interval' => $price->recurring->interval ?? null,
                'current_period_end' => $sub->current_period_end ?? null,
                'crm_invoice_id' => $crmInvoiceId,
                'crm_invoice_status' => $crmInvoiceStatus,
                'crm_invoice_number' => $crmInvoiceNumber,
            ];
        }

        foreach ($scheduledInvoices as $invoice) {
            $crmInvoiceId = $invoice->metadata->crm_invoice_id ?? null;
            $crmInvoiceStatus = null;
            $crmInvoiceNumber = null;
            if ($crmInvoiceId) {
                $map = $invoiceMap[(int) $crmInvoiceId] ?? null;
                $crmInvoiceStatus = $map['status'] ?? null;
                $crmInvoiceNumber = $map['invoicenum'] ?? null;
            }

            $result[] = [
                'id' => $invoice->id,
                'status' => 'scheduled',
                'price_id' => null,
                'amount' => $invoice->amount_due ?? null,
                'currency' => $invoice->currency ?? null,
                'interval' => 'one_time',
                'current_period_end' => $invoice->automatically_finalizes_at ?? null,
                'crm_invoice_id' => $crmInvoiceId,
                'crm_invoice_status' => $crmInvoiceStatus,
                'crm_invoice_number' => $crmInvoiceNumber,
            ];
        }

        foreach ($oneTimeInvoices as $invoice) {
            $crmInvoiceId = $invoice->metadata->crm_invoice_id ?? null;
            $crmInvoiceStatus = null;
            $crmInvoiceNumber = null;
            if ($crmInvoiceId) {
                $map = $invoiceMap[(int) $crmInvoiceId] ?? null;
                $crmInvoiceStatus = $map['status'] ?? null;
                $crmInvoiceNumber = $map['invoicenum'] ?? null;
            }

            $paidAt = $invoice->status_transitions->paid_at ?? null;
            $timestamp = $paidAt ?? $invoice->automatically_finalizes_at ?? $invoice->created ?? null;

            $result[] = [
                'id' => $invoice->id,
                'status' => $invoice->status ?? 'paid',
                'price_id' => null,
                'amount' => $invoice->amount_paid ?? $invoice->amount_due ?? null,
                'currency' => $invoice->currency ?? null,
                'interval' => 'one_time',
                'current_period_end' => $timestamp,
                'crm_invoice_id' => $crmInvoiceId,
                'crm_invoice_status' => $crmInvoiceStatus,
                'crm_invoice_number' => $crmInvoiceNumber,
            ];
        }

        $this->jsonResponse(['data' => $result]);
    }

    public function cancelSubscriptionOrInvoice()
    {
        $payload = $this->getJsonBody();
        $stripeId = trim((string) ($payload['stripe_id'] ?? _post('stripe_id')));
        $kind = trim((string) ($payload['kind'] ?? _post('kind')));

        if ($stripeId === '') {
            $this->jsonResponse(['error' => 'stripe_id is required'], 400);
        }

        $isInvoice = false;
        if ($kind !== '') {
            $isInvoice = $kind === 'invoice';
        } elseif (str_starts_with($stripeId, 'in_')) {
            $isInvoice = true;
        }

        try {
            $stripe = $this->getStripeClient();
            if ($isInvoice) {
                $invoice = null;
                try {
                    $invoice = $stripe->invoices->retrieve($stripeId, []);
                } catch (\Throwable $e) {
                    $invoice = null;
                }

                try {
                    $stripe->invoices->delete($stripeId, []);
                } catch (\Throwable $e) {
                    $stripe->invoices->voidInvoice($stripeId, []);
                }

                if ($invoice && !empty($invoice->customer)) {
                    $metadata = $invoice->metadata ?? null;
                    $crmInvoiceId = $metadata->crm_invoice_id ?? null;
                    try {
                        $items = $stripe->invoiceItems->all([
                            'customer' => $invoice->customer,
                            'pending' => true,
                            'limit' => 100,
                        ]);
                        foreach ($items->data as $item) {
                            $itemMeta = $item->metadata ?? null;
                            $itemCrmId = $itemMeta->crm_invoice_id ?? null;
                            if ($crmInvoiceId && $itemCrmId === $crmInvoiceId) {
                                $stripe->invoiceItems->delete($item->id, []);
                                continue;
                            }
                            if ($item->invoice === $stripeId) {
                                $stripe->invoiceItems->delete($item->id, []);
                            }
                        }
                    } catch (\Throwable $e) {
                        // best effort cleanup
                    }
                }
            } else {
                $stripe->subscriptions->update($stripeId, [
                    'cancel_at_period_end' => true,
                ]);
            }
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        $this->jsonResponse(['ok' => true]);
    }

    public function cancelSubscriptionOrInvoiceImmediate()
    {
        $payload = $this->getJsonBody();
        $stripeId = trim((string) ($payload['stripe_id'] ?? _post('stripe_id')));
        $kind = trim((string) ($payload['kind'] ?? _post('kind')));

        if ($stripeId === '') {
            $this->jsonResponse(['error' => 'stripe_id is required'], 400);
        }

        $isInvoice = false;
        if ($kind !== '') {
            $isInvoice = $kind === 'invoice';
        } elseif (str_starts_with($stripeId, 'in_')) {
            $isInvoice = true;
        }

        try {
            $stripe = $this->getStripeClient();
            if ($isInvoice) {
                $invoice = null;
                try {
                    $invoice = $stripe->invoices->retrieve($stripeId, []);
                } catch (\Throwable $e) {
                    $invoice = null;
                }

                try {
                    $stripe->invoices->delete($stripeId, []);
                } catch (\Throwable $e) {
                    $stripe->invoices->voidInvoice($stripeId, []);
                }

                if ($invoice && !empty($invoice->customer)) {
                    $metadata = $invoice->metadata ?? null;
                    $crmInvoiceId = $metadata->crm_invoice_id ?? null;
                    try {
                        $items = $stripe->invoiceItems->all([
                            'customer' => $invoice->customer,
                            'pending' => true,
                            'limit' => 100,
                        ]);
                        foreach ($items->data as $item) {
                            $itemMeta = $item->metadata ?? null;
                            $itemCrmId = $itemMeta->crm_invoice_id ?? null;
                            if ($crmInvoiceId && $itemCrmId === $crmInvoiceId) {
                                $stripe->invoiceItems->delete($item->id, []);
                                continue;
                            }
                            if ($item->invoice === $stripeId) {
                                $stripe->invoiceItems->delete($item->id, []);
                            }
                        }
                    } catch (\Throwable $e) {
                        // best effort cleanup
                    }
                }
            } else {
                $stripe->subscriptions->cancel($stripeId, []);
            }
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        $this->jsonResponse(['ok' => true]);
    }

    public function listPaymentMethods()
    {
        $payload = $this->getJsonBody();
        $stripeCustomerId = trim((string) ($payload['stripe_customer_id'] ?? _post('stripe_customer_id')));

        if ($stripeCustomerId === '') {
            $this->jsonResponse(['error' => 'stripe_customer_id is required'], 400);
        }

        try {
            $stripe = $this->getStripeClient();
            $methods = $stripe->paymentMethods->all([
                'customer' => $stripeCustomerId,
                'type' => 'card',
            ]);
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        $savedMap = [];
        if ($this->tableExists('credit_plan_payment_method')) {
            $ids = [];
            foreach ($methods->data as $pm) {
                $ids[] = $pm->id;
            }
            if (!empty($ids)) {
                $rows = ORM::for_table('credit_plan_payment_method')
                    ->select('payment_method')
                    ->where_in('payment_method', $ids)
                    ->find_many();
                foreach ($rows as $row) {
                    $savedMap[$row->payment_method] = true;
                }
            }
        }

        $result = [];
        foreach ($methods->data as $pm) {
            $card = $pm->card ?? null;
            $result[] = [
                'id' => $pm->id,
                'brand' => $card->brand ?? null,
                'last4' => $card->last4 ?? null,
                'exp_month' => $card->exp_month ?? null,
                'exp_year' => $card->exp_year ?? null,
                'name' => $pm->billing_details->name ?? null,
                'email' => $pm->billing_details->email ?? null,
                'saved' => !empty($savedMap[$pm->id]),
            ];
        }

        $this->jsonResponse(['data' => $result]);
    }

    public function getDefaultPaymentMethod()
    {
        $payload = $this->getJsonBody();
        $stripeCustomerId = trim((string) ($payload['stripe_customer_id'] ?? _post('stripe_customer_id')));

        if ($stripeCustomerId === '') {
            $this->jsonResponse(['error' => 'stripe_customer_id is required'], 400);
        }

        try {
            $stripe = $this->getStripeClient();
            $customer = $stripe->customers->retrieve($stripeCustomerId, []);
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        $defaultPaymentMethod = $customer->invoice_settings->default_payment_method ?? null;
        $this->jsonResponse(['payment_method_id' => $defaultPaymentMethod]);
    }

    public function savePaymentMethod()
    {
        $payload = $this->getJsonBody();
        $crmCustomerId = (int) ($payload['crm_customer_id'] ?? _post('crm_customer_id'));
        $stripeCustomerId = trim((string) ($payload['stripe_customer_id'] ?? _post('stripe_customer_id')));
        $paymentMethodId = trim((string) ($payload['payment_method_id'] ?? _post('payment_method_id')));
        $estadoId = (int) ($payload['estado_id'] ?? _post('estado_id') ?? 1);
        $useCreditCard = (int) ($payload['use_credit_card'] ?? _post('use_credit_card') ?? 2);
        $configuracionStripeId = (int) ($payload['configuracion_stripe_id'] ?? _post('configuracion_stripe_id') ?? 0);

        if ($crmCustomerId <= 0 || $stripeCustomerId === '' || $paymentMethodId === '') {
            $this->jsonResponse(['error' => 'crm_customer_id, stripe_customer_id and payment_method_id are required'], 400);
        }

        if (!$this->tableExists('credit_plan_payment_method')) {
            $this->jsonResponse(['error' => 'credit_plan_payment_method table missing'], 400);
        }

        $existing = ORM::for_table('credit_plan_payment_method')
            ->where('crm_account_id', $crmCustomerId)
            ->where('payment_method', $paymentMethodId)
            ->find_one();

        if ($existing) {
            $this->jsonResponse(['ok' => true, 'id' => $existing->id]);
        }

        try {
            $stripe = $this->getStripeClient();
            $pm = $stripe->paymentMethods->retrieve($paymentMethodId, []);
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        $card = $pm->card ?? null;
        $cardholderName = trim((string) ($payload['titular_tarjeta'] ?? ''));
        $stripeName = trim((string) ($pm->billing_details->name ?? ''));
        if ($stripeName === '' && $cardholderName !== '') {
            $stripeName = $cardholderName;
        }
        $uuid = UUIDHelper::generateUniqueId();
        $uuidEnc = UUIDHelper::convertToBase64($uuid);
        $gatewayId = $configuracionStripeId ?: $this->getStripeGatewayId();
        $gatewayConfig = null;
        if ($gatewayId) {
            $gatewayConfig = ORM::for_table('sys_pg')
                ->where('id', $gatewayId)
                ->find_one();
        }
        $gatewayPayload = null;
        if ($gatewayConfig) {
            $gatewayPayload = [
                'id_stripe' => (string) $gatewayId,
                'pk_stripe' => (string) ($gatewayConfig->value ?? ''),
                'sk_stripe' => (string) ($gatewayConfig->c1 ?? ''),
            ];
        }
        if (!in_array($estadoId, [1, 2], true)) {
            $estadoId = 1;
        }
        if (!in_array($useCreditCard, [1, 2], true)) {
            $useCreditCard = 2;
        }

        $data = [
            'crm_account_id' => $crmCustomerId,
            'tipo_cuenta' => 3,
            'tipo_pasarela_pago' => 'Stripe',
            'titular_tarjeta' => $stripeName,
            'numero_tarjeta' => $card->last4 ?? '',
            'exp_month' => $card->exp_month ?? '',
            'exp_year' => $card->exp_year ?? '',
            'estado_id' => $estadoId,
            'use_credit_card' => $useCreditCard,
            'banco_metodo_pago' => $card->brand ?? '',
            'nombre_banco_metodo_pago' => $card->brand ?? '',
            'email_metodo_pago' => $pm->billing_details->email ?? '',
            'payment_method' => $pm->id,
            'customer_id' => $stripeCustomerId,
            'verificado' => 1,
            'tipo_verificacion_micro_deposito' => '',
            'UUID' => $uuid,
            'UUID_ENC' => $uuidEnc,
            'configuracion_stripe_id' => $gatewayId,
            'configuracion_pasarela_pago' => $gatewayPayload ? json_encode($gatewayPayload) : '',
        ];

        $allowed = $this->getTableColumns('credit_plan_payment_method');
        $filtered = [];
        foreach ($data as $key => $value) {
            if (in_array($key, $allowed, true)) {
                $filtered[$key] = $value;
            }
        }

        $row = ORM::for_table('credit_plan_payment_method')->create();
        foreach ($filtered as $key => $value) {
            $row->$key = $value;
        }
        $row->save();

        if ($useCreditCard === 1) {
            try {
                $stripe->customers->update($stripeCustomerId, [
                    'invoice_settings' => [
                        'default_payment_method' => $pm->id,
                    ],
                ]);
            } catch (\Throwable $e) {
                // Best effort; default can be set later.
            }
        }

        $this->jsonResponse(['ok' => true, 'id' => $row->id]);
    }

    public function createBankAccountSetup()
    {
        $payload = $this->getJsonBody();
        $crmCustomerId = (int) ($payload['crm_customer_id'] ?? _post('crm_customer_id'));
        $accountNumber = trim((string) ($payload['account_number'] ?? _post('account_number')));
        $routingNumber = trim((string) ($payload['routing_number'] ?? _post('routing_number')));
        $accountHolderType = trim((string) ($payload['account_holder_type'] ?? _post('account_holder_type')));
        $email = trim((string) ($payload['email'] ?? _post('email')));
        $name = trim((string) ($payload['name'] ?? _post('name')));
        $gatewayId = (int) ($payload['configuracion_stripe_id'] ?? _post('configuracion_stripe_id') ?? 0);

        if ($crmCustomerId <= 0) {
            $this->jsonResponse(['error' => 'crm_customer_id is required'], 400);
        }

        if ($accountNumber === '' || $routingNumber === '' || $accountHolderType === '') {
            $this->jsonResponse(['error' => 'account_number, routing_number and account_holder_type are required'], 400);
        }

        try {
            $stripe = $this->getStripeClientForGateway($gatewayId);
            $stripeCustomerId = $this->getOrCreateStripeCustomerId($crmCustomerId, $email, $name, $stripe);

            $setupIntent = $stripe->setupIntents->create([
                'customer' => $stripeCustomerId,
                'payment_method_data' => [
                    'type' => 'us_bank_account',
                    'us_bank_account' => [
                        'account_number' => $accountNumber,
                        'routing_number' => $routingNumber,
                        'account_holder_type' => $accountHolderType,
                    ],
                    'billing_details' => [
                        'name' => $name,
                        'email' => $email,
                    ],
                ],
                'payment_method_types' => ['us_bank_account'],
                'confirm' => true,
                'mandate_data' => [
                    'customer_acceptance' => [
                        'type' => 'online',
                        'online' => [
                            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1',
                            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'php-client',
                        ],
                    ],
                ],
            ]);
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        $microdepositType = null;
        if (!empty($setupIntent->next_action) && !empty($setupIntent->next_action->verify_with_microdeposits)) {
            $microdepositType = $setupIntent->next_action->verify_with_microdeposits->microdeposit_type ?? null;
        }

        $this->jsonResponse([
            'stripe_customer_id' => $stripeCustomerId,
            'setup_intent' => $setupIntent->id,
            'payment_method' => $setupIntent->payment_method ?? null,
            'status' => $setupIntent->status ?? null,
            'tipo_verificacion_micro_deposito' => $microdepositType,
            'verified' => ($setupIntent->status ?? '') === 'succeeded',
        ]);
    }

    public function createSetupIntent()
    {
        $payload = $this->getJsonBody();
        $stripeCustomerId = trim((string) ($payload['stripe_customer_id'] ?? _post('stripe_customer_id')));

        if ($stripeCustomerId === '') {
            $this->jsonResponse(['error' => 'stripe_customer_id is required'], 400);
        }

        try {
            $stripe = $this->getStripeClient();
            $setupIntent = $stripe->setupIntents->create([
                'customer' => $stripeCustomerId,
                'payment_method_types' => ['card'],
            ]);
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        $this->jsonResponse([
            'client_secret' => $setupIntent->client_secret,
        ]);
    }

    public function setDefaultPaymentMethod()
    {
        $payload = $this->getJsonBody();
        $stripeCustomerId = trim((string) ($payload['stripe_customer_id'] ?? _post('stripe_customer_id')));
        $paymentMethodId = trim((string) ($payload['payment_method_id'] ?? _post('payment_method_id')));

        if ($stripeCustomerId === '' || $paymentMethodId === '') {
            $this->jsonResponse(['error' => 'stripe_customer_id and payment_method_id are required'], 400);
        }

        try {
            $stripe = $this->getStripeClient();
            $pm = $stripe->paymentMethods->retrieve($paymentMethodId, []);
            $pmCustomer = $pm->customer ?? null;
            if ($pmCustomer && $pmCustomer !== $stripeCustomerId) {
                $this->jsonResponse(['error' => 'Payment method belongs to another customer'], 400);
            }
            if (!$pmCustomer) {
                $stripe->paymentMethods->attach($paymentMethodId, [
                    'customer' => $stripeCustomerId,
                ]);
            }
            $stripe->customers->update($stripeCustomerId, [
                'invoice_settings' => [
                    'default_payment_method' => $paymentMethodId,
                ],
            ]);
        } catch (\Throwable $e) {
            $message = $e->getMessage();
            if (stripos($message, 'must be verified') !== false) {
                $this->jsonResponse([
                    'error' => $message,
                    'code' => 'bank_account_unverified',
                ], 400);
            }
            $this->jsonResponse(['error' => $message], 400);
        }

        $this->jsonResponse(['ok' => true]);
    }

    public function createSubscription()
    {
        $payload = $this->getJsonBody();
        $stripeCustomerId = trim((string) ($payload['stripe_customer_id'] ?? _post('stripe_customer_id')));
        $priceId = trim((string) ($payload['price_id'] ?? _post('price_id')));
        $crmInvoiceId = (int) ($payload['crm_invoice_id'] ?? _post('crm_invoice_id'));
        $frequency = trim((string) ($payload['frequency'] ?? _post('frequency')));
        $chargeType = trim((string) ($payload['charge_type'] ?? _post('charge_type') ?? 'recurring'));
        $chargeDate = trim((string) ($payload['charge_date'] ?? _post('charge_date')));
        $chargeTime = trim((string) ($payload['charge_time'] ?? _post('charge_time')));
        $customAmount = (float) ($payload['custom_amount'] ?? _post('custom_amount') ?? 0);

        if ($stripeCustomerId === '') {
            $this->jsonResponse(['error' => 'stripe_customer_id is required'], 400);
        }

        $items = [];
        $metadata = [];

        if ($crmInvoiceId > 0) {
            $invoice = ORM::for_table('sys_invoices')->find_one($crmInvoiceId);
            if (!$invoice) {
                $this->jsonResponse(['error' => 'CRM invoice not found'], 404);
            }

            $amount = (float) ($invoice->total ?? 0);
            if ($amount <= 0) {
                $this->jsonResponse(['error' => 'Invoice amount must be greater than zero'], 400);
            }

            $currency = (string) ($invoice->currency_iso_code ?? $invoice->currency ?? 'usd');
            $currency = strtolower($currency);
            if (strlen($currency) !== 3) {
                $currency = 'usd';
            }

            $invoiceNumber = trim((string) (($invoice->invoicenum ?? '') . ($invoice->cn ?? '')));
            if ($invoiceNumber === '') {
                $invoiceNumber = (string) $crmInvoiceId;
            }

            [$interval, $intervalCount] = $this->mapFrequency($frequency);

            $metadata['crm_invoice_id'] = (string) $crmInvoiceId;
            if ($customAmount > 0) {
                $amount = $customAmount;
                $metadata['crm_custom_amount'] = (string) $customAmount;
            } elseif ($customAmount < 0) {
                $this->jsonResponse(['error' => 'Custom amount must be greater than zero'], 400);
            }
        } elseif ($priceId !== '') {
            $items[] = [
                'price' => $priceId,
            ];
        } else {
            $this->jsonResponse(['error' => 'price_id or crm_invoice_id is required'], 400);
        }

        try {
            $stripe = $this->getStripeClient();

            if ($crmInvoiceId > 0 && $chargeType === 'one_time') {
                if ($chargeDate === '') {
                    $this->jsonResponse(['error' => 'charge_date is required for one_time'], 400);
                }
                if ($chargeTime === '') {
                    $chargeTime = '00:00';
                }

                $timezone = new \DateTimeZone('America/New_York');
                $dateTime = \DateTimeImmutable::createFromFormat('Y-m-d H:i', $chargeDate . ' ' . $chargeTime, $timezone);
                $timestamp = $dateTime ? $dateTime->getTimestamp() : false;
                if ($timestamp === false || $timestamp <= time()) {
                    $this->jsonResponse(['error' => 'charge_date must be in the future'], 400);
                }

                $invoice = $stripe->invoices->create([
                    'customer' => $stripeCustomerId,
                    'collection_method' => 'charge_automatically',
                    'auto_advance' => true,
                    'automatically_finalizes_at' => $timestamp,
                    'metadata' => $metadata,
                ]);

                $stripe->invoiceItems->create([
                    'customer' => $stripeCustomerId,
                    'amount' => (int) round($amount * 100),
                    'currency' => $currency,
                    'description' => "CRM Invoice {$invoiceNumber} (ID {$crmInvoiceId})",
                    'metadata' => $metadata,
                    'invoice' => $invoice->id,
                ]);

                $this->jsonResponse([
                    'status' => 'scheduled',
                    'invoice_id' => $invoice->id,
                ]);
            }

            if ($crmInvoiceId > 0) {
                $product = $stripe->products->create([
                    'name' => "CRM Invoice {$invoiceNumber} (ID {$crmInvoiceId})",
                    'metadata' => $metadata,
                ]);

                $items[] = [
                    'price_data' => [
                        'currency' => $currency,
                        'unit_amount' => (int) round($amount * 100),
                        'recurring' => [
                            'interval' => $interval,
                            'interval_count' => $intervalCount,
                        ],
                        'product' => $product->id,
                    ],
                ];
            }
            $subscription = $stripe->subscriptions->create([
                'customer' => $stripeCustomerId,
                'items' => $items,
                'payment_behavior' => 'default_incomplete',
                'expand' => ['latest_invoice.payment_intent'],
                'metadata' => $metadata,
            ]);
        } catch (\Throwable $e) {
            $this->jsonResponse(['error' => $e->getMessage()], 400);
        }

        $paymentIntent = null;
        if (!empty($subscription->latest_invoice) && !empty($subscription->latest_invoice->payment_intent)) {
            $paymentIntent = $subscription->latest_invoice->payment_intent;
        }

        $this->jsonResponse([
            'subscription_id' => $subscription->id,
            'status' => $subscription->status,
            'client_secret' => $paymentIntent ? ($paymentIntent->client_secret ?? null) : null,
        ]);
    }

    public function handleStripeWebhook()
    {
        $payload = file_get_contents('php://input');
        $sigHeader = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';
        $secret = $_ENV['STRIPE_WEBHOOK_SECRET'] ?? getenv('STRIPE_WEBHOOK_SECRET') ?? '';
        $this->logStripeWebhook('incoming: sig=' . ($sigHeader !== '' ? 'yes' : 'no') . ' bytes=' . strlen($payload));
        $skipVerify = $this->shouldSkipWebhookVerification();

        if (!$skipVerify && $secret === '') {
            $this->logStripeWebhook('error: STRIPE_WEBHOOK_SECRET missing');
            http_response_code(400);
            exit;
        }

        require __DIR__ . '/../../stripe/init.php';

        try {
            if ($skipVerify) {
                $data = json_decode($payload, true);
                if (!is_array($data)) {
                    throw new \Exception('Invalid JSON payload.');
                }
                $event = \Stripe\Event::constructFrom($data);
            } else {
                $event = \Stripe\Webhook::constructEvent($payload, $sigHeader, $secret);
            }
        } catch (\Throwable $e) {
            $this->logStripeWebhook('error: signature verify failed: ' . $e->getMessage());
            http_response_code(400);
            exit;
        }

        try {
            switch ($event->type) {
                case 'invoice.payment_succeeded':
                    $this->handleInvoicePaymentSucceeded($event);
                    break;
                case 'invoice.payment_failed':
                    $this->handleInvoicePaymentFailed($event);
                    break;
                case 'customer.subscription.updated':
                    $this->handleSubscriptionUpdated($event);
                    break;
                case 'customer.subscription.deleted':
                    $this->handleSubscriptionDeleted($event);
                    break;
            }

            $this->logStripeWebhook($this->formatWebhookSummary($event));
        } catch (\Throwable $e) {
            $this->logStripeWebhook('error: webhook handler failed: ' . $e->getMessage());
            http_response_code(500);
            exit;
        }

        http_response_code(200);
        exit;
    }

    private function handleInvoicePaymentSucceeded($event): void
    {
        $subscriptionId = $event->data->object->subscription ?? null;
        if ($subscriptionId) {
            $this->updateSubscriptionStatus((string) $subscriptionId, 'Active');
        }

        $crmInvoiceId = $this->resolveCrmInvoiceIdFromEvent($event);
        if ($crmInvoiceId) {
            $amountPaid = (int) ($event->data->object->amount_paid ?? 0);
            $this->markCrmInvoicePaid($crmInvoiceId, $amountPaid);
            $this->recordCrmInvoiceTransaction($event, $crmInvoiceId, $amountPaid);
        }
    }

    private function handleInvoicePaymentFailed($event): void
    {
        $subscriptionId = $event->data->object->subscription ?? null;
        if ($subscriptionId) {
            $this->updateSubscriptionStatus((string) $subscriptionId, 'Past Due');
        }
    }

    private function handleSubscriptionUpdated($event): void
    {
        $subscriptionId = $event->data->object->id ?? null;
        $status = $event->data->object->status ?? null;
        if ($subscriptionId && $status) {
            $this->updateSubscriptionStatus((string) $subscriptionId, $this->mapStripeStatus((string) $status));
        }
    }

    private function handleSubscriptionDeleted($event): void
    {
        $subscriptionId = $event->data->object->id ?? null;
        if ($subscriptionId) {
            $this->updateSubscriptionStatus((string) $subscriptionId, 'Cancelled');
        }
    }

    private function updateSubscriptionStatus(string $stripeSubscriptionId, string $status): void
    {
        if (!$this->tableExists('subscriptions')) {
            return;
        }

        $column = $this->resolveSubscriptionIdColumn();
        if (!$column) {
            return;
        }

        $subscription = ORM::for_table('subscriptions')
            ->where($column, $stripeSubscriptionId)
            ->find_one();

        if ($subscription) {
            $subscription->status = $status;
            $subscription->save();
        }
    }

    private function mapStripeStatus(string $status): string
    {
        $map = [
            'active' => 'Active',
            'trialing' => 'Trialing',
            'past_due' => 'Past Due',
            'canceled' => 'Cancelled',
            'unpaid' => 'Unpaid',
            'incomplete' => 'Incomplete',
            'incomplete_expired' => 'Incomplete Expired',
        ];

        $key = strtolower($status);
        return $map[$key] ?? $status;
    }

    private function mapFrequency(?string $frequency): array
    {
        $frequency = strtolower((string) $frequency);
        switch ($frequency) {
            case 'weekly':
                return ['week', 1];
            case 'bi-weekly':
                return ['week', 2];
            case 'quarterly':
                return ['month', 3];
            case 'yearly':
                return ['year', 1];
            case 'monthly':
            default:
                return ['month', 1];
        }
    }

    private function resolveCrmInvoiceIdFromEvent($event): ?int
    {
        $invoice = $event->data->object ?? null;
        if (!$invoice) {
            return null;
        }

        $metadata = $invoice->metadata ?? null;
        $crmInvoiceId = $metadata->crm_invoice_id ?? null;
        if (!empty($crmInvoiceId)) {
            return (int) $crmInvoiceId;
        }

        $subscriptionDetails = $invoice->subscription_details ?? null;
        if (!empty($subscriptionDetails) && !empty($subscriptionDetails->metadata)) {
            $crmInvoiceId = $subscriptionDetails->metadata->crm_invoice_id ?? null;
            if (!empty($crmInvoiceId)) {
                return (int) $crmInvoiceId;
            }
        }

        $lines = $invoice->lines->data ?? null;
        if (is_array($lines)) {
            foreach ($lines as $line) {
                $lineMeta = $line->metadata ?? null;
                $crmInvoiceId = $lineMeta->crm_invoice_id ?? null;
                if (!empty($crmInvoiceId)) {
                    return (int) $crmInvoiceId;
                }
            }
        }

        $subscriptionId = $invoice->subscription ?? null;
        if (!$subscriptionId) {
            return null;
        }

        try {
            $stripe = $this->getStripeClient();
            $subscription = $stripe->subscriptions->retrieve($subscriptionId, []);
            $meta = $subscription->metadata ?? null;
            $crmInvoiceId = $meta->crm_invoice_id ?? null;
            return $crmInvoiceId ? (int) $crmInvoiceId : null;
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function formatWebhookSummary($event): string
    {
        $type = (string) ($event->type ?? 'unknown');
        $object = $event->data->object ?? null;

        $stripeInvoiceId = $object->id ?? null;
        $subscriptionId = $object->subscription ?? null;
        $amountPaid = $object->amount_paid ?? null;

        $crmInvoiceId = null;
        $metadata = $object->metadata ?? null;
        if ($metadata && !empty($metadata->crm_invoice_id)) {
            $crmInvoiceId = $metadata->crm_invoice_id;
        } elseif (!empty($object->subscription_details) && !empty($object->subscription_details->metadata)) {
            $crmInvoiceId = $object->subscription_details->metadata->crm_invoice_id ?? null;
        } elseif (!empty($object->parent) && !empty($object->parent->subscription_details)) {
            $crmInvoiceId = $object->parent->subscription_details->metadata->crm_invoice_id ?? null;
        }

        $parts = [
            'event=' . $type,
        ];
        if ($stripeInvoiceId) {
            $parts[] = 'invoice=' . $stripeInvoiceId;
        }
        if ($subscriptionId) {
            $parts[] = 'subscription=' . $subscriptionId;
        }
        if ($crmInvoiceId) {
            $parts[] = 'crm_invoice=' . $crmInvoiceId;
        }
        if ($amountPaid !== null) {
            $parts[] = 'amount_paid=' . $amountPaid;
        }

        return implode(' ', $parts);
    }

    private function markCrmInvoicePaid(int $invoiceId, int $amountPaidCents): void
    {
        $invoice = ORM::for_table('sys_invoices')->find_one($invoiceId);
        if (!$invoice) {
            return;
        }

        $amountPaid = $amountPaidCents > 0 ? ($amountPaidCents / 100) : 0;
        $currentCredit = (float) ($invoice->credit ?? 0);
        $total = (float) ($invoice->total ?? 0);

        $newCredit = $currentCredit + $amountPaid;
        if ($newCredit > $total && $total > 0) {
            $newCredit = $total;
        }

        $invoice->credit = $newCredit;
        $invoice->amount_last_paid = $amountPaid;
        $invoice->date_last_paid = date('Y-m-d H:i:s');
        $invoice->datepaid = date('Y-m-d H:i:s');
        $invoice->status = ($total > 0 && $newCredit >= $total) ? 'Paid' : 'Partially Paid';
        $invoice->save();
    }

    private function recordCrmInvoiceTransaction($event, int $invoiceId, int $amountPaidCents): void
    {
        if ($amountPaidCents <= 0) {
            return;
        }

        $invoice = ORM::for_table('sys_invoices')->find_one($invoiceId);
        if (!$invoice) {
            $this->logStripeWebhook("warn: transaction skipped, invoice not found id={$invoiceId}");
            return;
        }

        $stripeInvoice = $event->data->object ?? null;
        $paymentIntentId = $stripeInvoice->payment_intent ?? $stripeInvoice->charge ?? $stripeInvoice->id ?? null;
        if ($paymentIntentId) {
            $existing = ORM::for_table('sys_transactions')
                ->where('ref', $paymentIntentId)
                ->find_one();
            if ($existing) {
                $this->logStripeWebhook("info: transaction exists ref={$paymentIntentId} iid={$invoiceId}");
                return;
            }
        }

        $accountName = (string) ($invoice->account ?? '');
        $account = null;
        if ($accountName !== '') {
            $account = ORM::for_table('sys_accounts')
                ->where('account', $accountName)
                ->find_one();
        }
        if (!$account) {
            $account = ORM::for_table('sys_accounts')->order_by_asc('id')->find_one();
        }
        if (!$account) {
            $this->logStripeWebhook("warn: transaction skipped, account missing iid={$invoiceId}");
            return;
        }

        $amount = $amountPaidCents / 100;
        $currencyIso = (string) ($invoice->currency_iso_code ?? 'USD');
        $dateNow = date('Y-m-d H:i:s');

        $transaction = ORM::for_table('sys_transactions')->create();
        $transaction->account = $account->account ?? $accountName;
        $transaction->account_id = $account->id ?? 0;
        $transaction->type = 'Income';
        $transaction->payerid = $invoice->userid ?? 0;
        $transaction->amount = $amount;
        $transaction->method = 'Stripe AutoPay';
        $transaction->ref = $paymentIntentId ?: '';
        $transaction->description = 'Stripe autopay invoice #' . ($invoice->invoicenum ?? $invoiceId);
        $transaction->date = $dateNow;
        $transaction->dr = '0.00';
        $transaction->cr = $amount;
        $transaction->iid = $invoiceId;
        $transaction->status = 'Cleared';
        $transaction->currency_iso_code = $currencyIso;
        $transaction->currency = 0;
        $transaction->currency_symbol = '';
        $transaction->currency_rate = 1;
        $transaction->aid = $invoice->aid ?? 0;
        $transaction->staff_id = $invoice->aid ?? 0;
        $transaction->staff_id_pago = $invoice->aid ?? 0;
        $transaction->updated_at = $dateNow;

        try {
            $transaction->save();
            $this->logStripeWebhook("info: transaction saved ref={$paymentIntentId} iid={$invoiceId} amount={$amount}");
        } catch (\Throwable $e) {
            $this->logStripeWebhook('error: transaction save failed: ' . $e->getMessage());
        }
    }

    private function resolveStripeCustomerColumn(): ?string
    {
        $columns = $this->getTableColumns('crm_accounts');
        $candidates = ['customer_id_stripe', 'stripe_customer_id', 'stripe_customer', 'customer_id'];
        foreach ($candidates as $column) {
            if (in_array($column, $columns, true)) {
                return $column;
            }
        }

        return null;
    }

    private function isStripeCustomerId(string $value): bool
    {
        return $value !== '' && str_starts_with($value, 'cus_');
    }

    private function resolveSubscriptionIdColumn(): ?string
    {
        $columns = $this->getTableColumns('subscriptions');
        $candidates = ['stripe_subscription_id', 'stripe_subscription', 'stripe_id'];
        foreach ($candidates as $column) {
            if (in_array($column, $columns, true)) {
                return $column;
            }
        }

        return null;
    }

    private function getTableColumns(string $table): array
    {
        $pdo = ORM::get_db();
        $stmt = $pdo->query("DESCRIBE {$table}");
        if (!$stmt) {
            return [];
        }

        return array_column($stmt->fetchAll(\PDO::FETCH_ASSOC), 'Field');
    }

    private function tableExists(string $table): bool
    {
        $pdo = ORM::get_db();
        $stmt = $pdo->prepare('SHOW TABLES LIKE :table');
        $stmt->execute([':table' => $table]);
        return (bool) $stmt->fetchColumn();
    }

    private function getJsonBody(): array
    {
        $raw = file_get_contents('php://input');
        $data = json_decode($raw, true);
        return is_array($data) ? $data : [];
    }

    private function getStripeClient(): StripeClient
    {
        require __DIR__ . '/../../stripe/init.php';
        $secret = $this->resolveStripeSecretKey();
        return new StripeClient($secret);
    }

    private function getStripePublishableKey(): string
    {
        $gateway = ORM::for_table('sys_pg')
            ->where('processor', 'stripe')
            ->where('status', 'Active')
            ->order_by_desc('id')
            ->find_one();

        return $gateway->value ?? '';
    }

    private function getStripeGatewayId(): ?int
    {
        $gateway = ORM::for_table('sys_pg')
            ->where('processor', 'stripe')
            ->where('status', 'Active')
            ->order_by_desc('id')
            ->find_one();

        return $gateway ? (int) $gateway->id : null;
    }

    private function resolveStripeSecretKey(): string
    {
        $env = $_ENV['APP_ENV'] ?? getenv('APP_ENV') ?? 'prod';

        if ($env === 'local') {
            $secret = $_ENV['STRIPE_TEST_KEY'] ?? getenv('STRIPE_TEST_KEY') ?? '';
            $this->guardStripeKeyForEnv($secret);
            return $secret;
        }

        $secret = $_ENV['STRIPE_SECRET_KEY'] ?? getenv('STRIPE_SECRET_KEY') ?? '';
        $this->guardStripeKeyForEnv($secret);
        return $secret;
    }

    private function getStripeClientForGateway(?int $gatewayId): StripeClient
    {
        require __DIR__ . '/../../stripe/init.php';

        $secret = '';
        if ($gatewayId) {
            $gateway = ORM::for_table('sys_pg')
                ->where('id', $gatewayId)
                ->where('status', 'Active')
                ->find_one();
            $secret = trim((string) ($gateway->c1 ?? ''));
        }

        if ($secret === '') {
            $secret = $this->resolveStripeSecretKey();
        } else {
            $this->guardStripeKeyForEnv($secret);
        }

        return new StripeClient($secret);
    }

    private function getOrCreateStripeCustomerId(int $crmCustomerId, string $email, string $name, StripeClient $stripe): string
    {
        $contact = ORM::for_table('crm_accounts')->find_one($crmCustomerId);
        if (!$contact) {
            throw new \Exception('Customer not found.');
        }

        $stripeCustomerColumn = $this->resolveStripeCustomerColumn();
        if ($stripeCustomerColumn) {
            $existing = trim((string) ($contact->$stripeCustomerColumn ?? ''));
            if (!$this->isStripeCustomerId($existing)) {
                $existing = '';
            }
            if ($existing !== '') {
                try {
                    $stripe->customers->retrieve($existing, []);
                    return $existing;
                } catch (\Throwable $e) {
                    // fall through to create
                }
            }
        }

        $email = $email !== '' ? $email : trim((string) ($contact->email ?? ''));
        $name = $name !== '' ? $name : trim((string) ($contact->account ?? ''));
        $params = [];
        if ($email !== '') {
            $params['email'] = $email;
        }
        if ($name !== '') {
            $params['name'] = $name;
        }

        $customer = $stripe->customers->create($params);

        if ($stripeCustomerColumn) {
            $contact->$stripeCustomerColumn = $customer->id;
            $contact->save();
        }

        return $customer->id;
    }

    private function guardStripeKeyForEnv(string $key): void
    {
        $env = $_ENV['APP_ENV'] ?? getenv('APP_ENV') ?? 'prod';
        $key = (string) $key;

        if ($key === '') {
            throw new \Exception('Stripe secret key missing.');
        }

        if ($env !== 'prod' && str_starts_with($key, 'sk_live_')) {
            throw new \Exception("Blocked: {$env} cannot use Stripe LIVE key.");
        }

        if ($env === 'prod' && str_starts_with($key, 'sk_test_')) {
            throw new \Exception('Blocked: production cannot use Stripe TEST key.');
        }
    }

    private function logStripeWebhook(string $message): void
    {
        $logFile = __DIR__ . '/../../../storage/logs/stripe_webhook.log';
        $line = date('Y-m-d H:i:s') . ' ' . $message . PHP_EOL;
        @file_put_contents($logFile, $line, FILE_APPEND);
    }

    private function shouldSkipWebhookVerification(): bool
    {
        $env = $_ENV['APP_ENV'] ?? getenv('APP_ENV') ?? 'prod';
        $flag = $_ENV['STRIPE_WEBHOOK_SKIP_VERIFY'] ?? getenv('STRIPE_WEBHOOK_SKIP_VERIFY') ?? '';
        if (strtolower((string) $env) !== 'local') {
            return false;
        }
        return in_array(strtolower((string) $flag), ['1', 'true', 'yes'], true);
    }

    private function jsonResponse(array $data, int $status = 200): void
    {
        header('Content-Type: application/json');
        http_response_code($status);
        echo json_encode($data);
        exit;
    }
}
