<?php
namespace App\Controllers;

//use Aws\SecretsManager\SecretsManagerClient;
use App\Controllers\BaseController;
use App\Models\Invoice;
use App\Models\Currency;
use ORM;

class InvoiceController extends BaseController
{

    public function getInvoiceById($id)
    {
        header('Content-Type: application/json');
        global $config, $_L;

        if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
            http_response_code(405);
            echo json_encode([
                'status' => 'error',
                'message' => 'Method not allowed. Use GET.',
            ]);
            exit;
        }

        $id = (int) $id;

        if (empty($id)) {
            http_response_code(400);
            echo json_encode([
                'status' => 'error',
                'message' => 'Invalid invoice id',
            ]);
            exit;
        }

        try {
            $invoice = Invoice::find($id);

            if (!$invoice) {
                http_response_code(404);
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Invoice not found',
                ]);
                exit;
            }

            $invoiceData = method_exists($invoice, 'toArray') ? $invoice->toArray() : $invoice;

            $items = ORM::for_table('sys_invoiceitems')
                ->where('invoiceid', $id)
                ->order_by_asc('id')
                ->find_array();

            $transactions = ORM::for_table('sys_transactions')
                ->where('iid', $id)
                ->order_by_desc('id')
                ->find_array();

            $contact = null;
            $contactAddress = '';
            if (!empty($invoiceData['userid'])) {
                $contact = ORM::for_table('crm_accounts')->find_one($invoiceData['userid']);
                if ($contact && method_exists($contact, 'toArray')) {
                    $contact = $contact->toArray();
                }
                if (!empty($contact)) {
                    $contactAddress = getContactFormattedAddress($config, $contact);
                }
            }
            $contactCompany = '';
            if (is_array($contact) && !empty($contact['company'])) {
                $contactCompany = $contact['company'];
            } elseif (!empty($invoiceData['company_name'])) {
                $contactCompany = $invoiceData['company_name'];
            }

            $quote = null;
            if (!empty($invoiceData['quote_id']) && $invoiceData['quote_id'] != '0') {
                $quote = ORM::for_table('sys_quotes')->find_one($invoiceData['quote_id']);
                if ($quote && method_exists($quote, 'toArray')) {
                    $quote = $quote->toArray();
                }
            }

            $links = ORM::for_table('payment_links')
                ->where('id_invoice', $id)
                ->order_by_desc('id_payment')
                ->find_array();

            $db = ORM::get_db();

            $bankDetailsStmt = $db->prepare(
                'SELECT IDBAcc.*, IDSta.status AS name_status
                 FROM invoice_details_bank_account AS IDBAcc
                 LEFT JOIN invoice_detail_status AS IDSta ON IDSta.id = IDBAcc.payment_status
                 WHERE IDBAcc.invoices_id = :invoiceId'
            );
            $bankDetailsStmt->execute(['invoiceId' => $id]);
            $bankDetails = $bankDetailsStmt->fetchAll(\PDO::FETCH_ASSOC);

            $terminalStmt = $db->prepare(
                'SELECT
                    PT.*,
                    RT.serial_number_stripe
                    FROM payment_terminal PT
                    LEFT JOIN reader_terminal RT
                    ON RT.id_reader_stripe = PT.process_payment_intent_id
                    WHERE PT.id_invoice = :invoiceId;'
            );
            $terminalStmt->execute(['invoiceId' => $id]);
            $terminalPayments = $terminalStmt->fetchAll(\PDO::FETCH_ASSOC);

            $buildHistoryEntry = function ($type, array $data) {
                $historyDate = null;
                if (!empty($data['date_verify'])) {
                    $historyDate = strtotime($data['date_verify']);
                } elseif (!empty($data['date_create'])) {
                    $historyDate = strtotime($data['date_create']);
                }

                $error = '';
                if (!empty($data['mensaje_error'])) {
                    $error = $data['mensaje_error'];
                    if (!empty($data['codigo_error'])) {
                        $error .= ' (' . $data['codigo_error'] . ')';
                    }
                }

                return [
                    'source' => $type,
                    'method' => $data['method'] ?? ($data['account'] ?? ''),
                    'status' => $data['status'] ?? ($data['payment_status'] ?? ''),
                    'amount' => $data['amount'] ?? 0,
                    'date' => $historyDate,
                    'display_date' => $historyDate ? date('Y-m-d', $historyDate) : '',
                    'display_time' => $historyDate ? date('H:i', $historyDate) : '',
                    'staff' => $data['name_staff'] ?? '',
                    'error' => $error,
                    'serial_number_stripe' => $data['serial_number_stripe'] ?? '',
                    'link_url' => $data['url'] ?? '',
                    'link_id' => $data['id_payment'] ?? '',
                ];
            };

            $paymentHistory = [];

            foreach ($links as $link) {
                $paymentHistory[] = $buildHistoryEntry('link', [
                    'method' => !empty($link['type_payment']) ? $link['type_payment'] : 'Link',
                    'status' => $link['payment_status'] ?? '',
                    'amount' => $link['amount'] ?? 0,
                    'date_verify' => $link['date_verify'] ?? null,
                    'date_create' => $link['date_create'] ?? null,
                    'url' => $link['url'] ?? '',
                    'id_payment' => $link['id_payment'] ?? '',
                ]);
            }

            foreach ($bankDetails as $element) {
                $paymentHistory[] = $buildHistoryEntry('bank', [
                    'method' => $element['account'] ?? 'Bank Transfer',
                    'status' => $element['name_status'] ?? '',
                    'amount' => $element['amount'] ?? 0,
                    'date_verify' => $element['date_verify'] ?? null,
                    'date_create' => $element['date_create'] ?? null,
                    'name_staff' => $element['name_staff'] ?? '',
                    'mensaje_error' => $element['mensaje_error'] ?? '',
                    'codigo_error' => $element['codigo_error'] ?? '',
                ]);
            }

            foreach ($terminalPayments as $element) {
                $paymentHistory[] = $buildHistoryEntry('terminal', [
                    'method' => $element['account'] ?? 'POS Terminal',
                    'status' => $element['payment_status'] ?? '',
                    'amount' => $element['amount'] ?? 0,
                    'date_verify' => $element['date_verify'] ?? null,
                    'date_create' => $element['date_create'] ?? null,
                    'name_staff' => $element['name_staff'] ?? '',
                    'mensaje_error' => $element['mensaje_error'] ?? '',
                    'codigo_error' => $element['codigo_error'] ?? '',
                    'serial_number_stripe' => $element['serial_number_stripe'] ?? '',
                ]);
            }

            usort($paymentHistory, function ($a, $b) {
                return ($b['date'] ?? 0) <=> ($a['date'] ?? 0);
            });


            $invoiceDesc = $invoiceData["invoicenum"] . $invoiceData["cn"] . " " . "Payment";

            $response = [
                'status' => 'success',
                'id' => $invoiceData['cn'],
                'invoice' => $invoiceData,
                'config' => [
                    'tax_system' => $config['tax_system'] ?? '',
                    'invoicing_allow_staff_selection_for_each_item' => $config['invoicing_allow_staff_selection_for_each_item'] ?? '',
                    'invoice_receipt_number' => $config['invoice_receipt_number'] ?? '',
                    'show_business_number' => $config['show_business_number'] ?? '',
                    'fax_field' => $config['fax_field'] ?? '',
                    'label_tax_number' => function_exists('sp_config_or_default')
                        ? sp_config_or_default('label_tax_number', $_L['Vat number'] ?? 'Vat number')
                        : ($_L['Vat number'] ?? 'Vat number'),
                    'logo_default' => $config['logo_default'] ?? '',
                    'company_name' => $config['CompanyName'] ?? '',
                    'company_address' => $config['caddress'] ?? '',
                    'vat_number' => $config['vat_number'] ?? '',
                ],
                'contact' => $contact ? [
                    'account' => $contact['account'] ?? '',
                    'company' => $contactCompany,
                    'business_number' => $contact['business_number'] ?? '',
                    'phone' => $contact['phone'] ?? '',
                    'fax' => $contact['fax'] ?? '',
                    'email' => $contact['email'] ?? '',
                    'formatted_address' => $contactAddress,
                ] : null,
                'formatted' => [
                    'date' => !empty($invoiceData['date']) ? date($config['df'], strtotime($invoiceData['date'])) : '',
                    'due' => !empty($invoiceData['duedate']) ? date($config['df'], strtotime($invoiceData['duedate'])) : '',
                ],
                'description' => $invoiceDesc,
                'items' => $items,
                'transactions' => $transactions,
                'quote' => $quote,
                'paymentLinks' => $links,
                'bankAccounts' => $bankDetails,
                'terminalPayments' => $terminalPayments,
                'paymentHistory' => $paymentHistory,
            ];

            http_response_code(200);
            echo json_encode($response, JSON_UNESCAPED_UNICODE);
            exit;

        } catch (\Throwable $th) {
            http_response_code(500);
            echo json_encode([
                'status' => 'error',
                'message' => 'Unable to fetch invoice',
            ]);
        }
    }


    public function getInvoiceContext($id)
    {
        header('Content-Type: application/json');

        if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
            http_response_code(405);
            echo json_encode([
                'status' => 'error',
                'message' => 'Method not allowed. Use GET.',
            ]);
            exit;
        }

        $invoiceId = (int) $id;

        if (empty($invoiceId)) {
            http_response_code(400);
            echo json_encode([
                'status' => 'error',
                'message' => 'Invalid invoice id',
            ]);
            exit;
        }

        try {
            $invoice = Invoice::find($invoiceId);

            if (!$invoice) {
                http_response_code(404);
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Invoice not found',
                ]);
                return;
            }

            $currencyISOCode = $invoice['currency_iso_code'];

            $accounts = ORM::for_table('sys_accounts')
                ->select_many('id', 'account')
                ->find_array();

            $paymentMethods = ORM::for_table('sys_pmethods')
                ->order_by_asc('sorder')
                ->find_array();

            $categories = ORM::for_table('sys_cats')
                ->where('type', 'Income')
                ->order_by_asc('sorder')
                ->find_array();

            $db = ORM::get_db();

            $accountMethodsStmt = $db->prepare(
                'SELECT CPPMet.*, Spg.status AS gateway_status
                 FROM credit_plan_payment_method AS CPPMet
                 LEFT JOIN sys_pg AS Spg ON Spg.id = CPPMet.configuracion_stripe_id
                 WHERE CPPMet.crm_account_id = :crm_account_id
                   AND CPPMet.tipo_cuenta IN (1,2)
                   AND CPPMet.estado_id = 1
                   AND Spg.status = "Active"'
            );
            
            $accountMethodsStmt->execute(['crm_account_id' => $invoice['userid'] ?? 0]);
            $accountMethods = $accountMethodsStmt->fetchAll(\PDO::FETCH_ASSOC);

            $paymentGateways = ORM::for_table('sys_pg')
                ->where('status', 'Active')
                ->find_array();

            $readerTerminals = ORM::for_table('reader_terminal')
                ->select_many(
                    'reader_terminal.id',
                    'reader_terminal.identification',
                    'reader_terminal.serial_number_stripe',
                    'reader_terminal.stripe_configuration_id'
                )
                ->join('sys_pg', ['sys_pg.id', '=', 'reader_terminal.stripe_configuration_id'])
                ->where('sys_pg.status', 'Active')
                ->find_array();

            $staffs = ORM::for_table('sys_users')
                ->select_many('id', 'fullname')
                ->where('status', 'Active')
                ->find_array();

            http_response_code(200);
            echo json_encode([
                'status' => 'success',
                'staffs' => $staffs,
                'currencyISOCode' => $currencyISOCode,
                'accounts' => $accounts,
                'paymentMethods' => $paymentMethods,
                'categories' => $categories,
                'accountMethods' => $accountMethods,
                'paymentGateways' => $paymentGateways,
                'readerTerminals' => $readerTerminals,
            ], JSON_UNESCAPED_UNICODE);
            exit;

        } catch (\Throwable $th) {
            http_response_code(500);
            echo json_encode([
                'status' => 'error',
                'message' => 'Unable to fetch invoice params',
            ]);
            exit;
        }
    }

}
