<?php
namespace App\Controllers;

use App\Services\PbxService;
use App\Models\Contact;
use ORM;

class PbxController
{
    private $PBX_BASE_URL;
    private $PBX_EMAIL;
    private $PBX_PASSWORD;

    public function __construct()
    {
        $this->PBX_BASE_URL   = $_ENV['PBX_BASE_URL'];
        $this->PBX_EMAIL      = $_ENV['PBX_EMAIL'];
        $this->PBX_PASSWORD   = $_ENV['PBX_PASSWORD'];
    }

    private function pbxConfigured(): bool
    {
        return !empty($this->PBX_BASE_URL) && !empty($this->PBX_EMAIL) && !empty($this->PBX_PASSWORD);
    }

    private function failNotConfigured(): bool
    {
        if ($this->pbxConfigured()) {
            return true;
        }
        header('Content-Type: application/json');
        http_response_code(500);
        echo json_encode(['error' => 'PBX environment variables are missing or not set.']);
        return false;
    }

    private function getPbxService(): PbxService
    {
        return new PbxService([
            'base_url' => $this->PBX_BASE_URL,
            'email' => $this->PBX_EMAIL,
            'password' => $this->PBX_PASSWORD
        ]);
    }

    private function encryptToken($token)
    {
        $key = $_ENV['PBX_ENCRYPTION_KEY'] ?? null;
        if (!$key) {
            throw new \Exception('Encryption key for PBX tokens is not configured.');
        }

        $encryptionKey = hash('sha256', $key, true);
        $iv = random_bytes(16);
        $cipher = openssl_encrypt($token, 'aes-256-gcm', $encryptionKey, OPENSSL_RAW_DATA, $iv, $tag);

        if ($cipher === false) {
            throw new \Exception('Failed to encrypt PBX token.');
        }

        return base64_encode($iv . $tag . $cipher);
    }

    private function decryptToken($encoded)
    {
        $key = $_ENV['PBX_ENCRYPTION_KEY'] ?? null;
        if (!$key) {
            throw new \Exception('Encryption key for PBX tokens is not configured.');
        }

        $decoded = base64_decode($encoded, true);
        if ($decoded === false || strlen($decoded) < 32) {
            throw new \Exception('Stored PBX token is invalid.');
        }

        $iv = substr($decoded, 0, 16);
        $tag = substr($decoded, 16, 16);
        $cipher = substr($decoded, 32);

        $encryptionKey = hash('sha256', $key, true);
        $plain = openssl_decrypt($cipher, 'aes-256-gcm', $encryptionKey, OPENSSL_RAW_DATA, $iv, $tag);
        if ($plain === false) {
            throw new \Exception('Failed to decrypt PBX token.');
        }

        return $plain;
    }

    private function getStoredPbxConnection($clientId)
    {
        return ORM::for_table('pbx_connections')
            ->where('client_id', $clientId)
            ->where('status', 'active')
            ->order_by_desc('last_sync')
            ->order_by_desc('id')
            ->find_one();
    }

    private function storePbxConnection($clientId, $token, $expiresAt = null)
    {
        $clientId = (int) $clientId;
        $pdo = ORM::get_db();
        $finalStoredValue = $this->encryptToken($token);

        $sql = "INSERT INTO pbx_connections 
                (client_id, bearer_token, expires_at, status, last_sync) 
                VALUES 
                (:client_id, :bearer_token, :expires_at, 'active', NOW())
                ON DUPLICATE KEY UPDATE 
                bearer_token = :bearer_token,
                expires_at = :expires_at,
                status = 'active',
                last_sync = NOW()";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            'client_id'     => $clientId,
            'bearer_token'  => $finalStoredValue,
            'expires_at'    => $expiresAt
        ]);
    }

    private function acquirePbxSyncLock(): bool
    {
        $pdo = ORM::get_db();
        $stmt = $pdo->query("SELECT GET_LOCK('pbx_cdr_sync', 0) AS got_lock");
        $row = $stmt ? $stmt->fetch(\PDO::FETCH_ASSOC) : null;
        return isset($row['got_lock']) && (int) $row['got_lock'] === 1;
    }

    private function releasePbxSyncLock(): void
    {
        $pdo = ORM::get_db();
        $pdo->query("SELECT RELEASE_LOCK('pbx_cdr_sync')");
    }

    // -------------------------
    // 1) Autenticarse con PBX
    // POST /api/pbx/authenticate
    // -------------------------
    public function authenticate()
    {
        header('Content-Type: application/json');
        if (!$this->failNotConfigured()) {
            exit;
        }
        
        $clientId = 1;

        try {
            $pbxService = $this->getPbxService();
            $authResponse = $pbxService->authenticate();

            // Store the encrypted token
            $this->storePbxConnection($clientId, $authResponse['token'], $authResponse['expires_at']);

            error_log("Autenticación PBX exitosa para cliente: " . $clientId);
            error_log("Token obtenido: " . substr($authResponse['token'], 0, 50) . "...");
            error_log("JSON completo auth: " . json_encode($authResponse));

            http_response_code(200);
            echo json_encode([
                'success' => true,
                'token' => $authResponse['token'],
                'expires_at' => $authResponse['expires_at'],
                'message' => 'Authentication successful'
            ]);
            exit;
        } catch (\Exception $e) {
            error_log("Error en autenticación PBX: " . $e->getMessage());
            error_log("Stack trace auth: " . $e->getTraceAsString());
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
            exit;
        }
    }

    // -------------------------
    // 2) Obtener datos CDR
    // POST /api/pbx/cdr
    // Body JSON: { "month":"202512", "page":1, "limit":50, "filters":"" }
    // -------------------------
    public function getCdr()
    {
        // Logging para saber cuándo se llama
        error_log("PBX CDR Endpoint llamado - " . date('Y-m-d H:i:s'));
        error_log("Request data: " . json_encode(file_get_contents('php://input')));
        
        header('Content-Type: application/json');
        if (!$this->failNotConfigured()) {
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true) ?: [];

        // Prepare CDR request parameters
        $params = [
            'month' => $data['month'] ?? date('Ym'),
            'page' => isset($data['page']) ? (int)$data['page'] : 1,
            'limit' => isset($data['limit']) ? (int)$data['limit'] : 50,
            'order' => $data['order'] ?? '-calldate',
            'filters' => $data['filters'] ?? '',
            'bypass_cache' => !empty($data['bypass_cache']),
            'page_delay_ms' => isset($data['page_delay_ms']) ? (int) $data['page_delay_ms'] : null,
            'max_pages' => isset($data['max_pages']) ? (int) $data['max_pages'] : null,
            'source' => $data['source'] ?? 'agg'
        ];

        if ($params['source'] === 'agg') {
            try {
                $pbxService = $this->getPbxService();
                if (!empty($data['sync'])) {
                    if (!$this->acquirePbxSyncLock()) {
                        http_response_code(200);
                        echo json_encode([
                            'success' => true,
                            'data' => [],
                            'skipped' => true,
                            'message' => 'PBX sync already in progress'
                        ]);
                        exit;
                    }

                    $clientId = 1;
                    $connection = $this->getStoredPbxConnection($clientId);
                    $token = null;

                    if (!$connection) {
                        $authResult = $pbxService->authenticate();
                        if (!$authResult || !isset($authResult['token'])) {
                            http_response_code(500);
                            echo json_encode(['error' => 'PBX authentication failed']);
                            exit;
                        }
                        $token = $authResult['token'];
                        $expiresAt = $authResult['expires_at'] ?? null;
                        $this->storePbxConnection($clientId, $token, $expiresAt);
                    } else {
                        $token = $this->decryptToken($connection->bearer_token);
                    }

                    $validToken = $pbxService->getValidToken($token);
                    if ($validToken !== $token) {
                        $this->storePbxConnection($clientId, $validToken);
                    }

                    $range = $pbxService->getDateRangeFromParams($params);
                    $syncParams = $params;
                    $syncParams['start_date'] = $range['start_date'];
                    $syncParams['end_date'] = $range['end_date'];

                    $lastSyncDate = $pbxService->getSyncDate();
                    if ($lastSyncDate) {
                        $nextDay = date('Y-m-d', strtotime($lastSyncDate . ' +1 day'));
                        if ($nextDay <= $syncParams['end_date']) {
                            $syncParams['start_date'] = $nextDay;
                        } else {
                            $syncParams = null;
                        }
                    }

                    try {
                        if ($syncParams) {
                            $pbxService->syncAggregatedCdr($validToken, $syncParams);
                        }
                    } finally {
                        $this->releasePbxSyncLock();
                    }
                }
                $cdrResponse = $pbxService->getAggregatedCdr($params);

                http_response_code(200);
                echo json_encode([
                    'success' => true,
                    'data' => $cdrResponse['data'],
                    'meta' => $cdrResponse['meta'],
                    'total' => $cdrResponse['total'],
                    'total_raw' => $cdrResponse['total_raw'] ?? null,
                    'timezone' => $cdrResponse['timezone'] ?? null,
                    'pagination' => $cdrResponse['pagination'] ?? null,
                    'params' => $params
                ]);
                exit;
            } catch (\Exception $e) {
                http_response_code(500);
                echo json_encode(['error' => $e->getMessage()]);
                exit;
            }
        }

        $clientId = 1;

        // Get stored connection or authenticate automatically
        $connection = $this->getStoredPbxConnection($clientId);
        $token = null;
        
        if (!$connection) {
            error_log("No hay conexión almacenada, autenticando automáticamente...");
            
            // Authenticate automatically
            try {
                $pbxService = $this->getPbxService();
                $authResult = $pbxService->authenticate();
                
                if (!$authResult || !isset($authResult['token'])) {
                    error_log("Error: Autenticación PBX falló");
                    http_response_code(500);
                    echo json_encode(['error' => 'PBX authentication failed']);
                    exit;
                }
                
                $token = $authResult['token'];
                $expiresAt = $authResult['expires_at'] ?? null;
                
                // Store the new token
                $this->storePbxConnection($clientId, $token, $expiresAt);
                error_log("Token PBX almacenado automáticamente para cliente: " . $clientId);
                
            } catch (\Exception $e) {
                error_log("Error en autenticacion automatica: " . $e->getMessage());
                http_response_code(500);
                echo json_encode(['error' => 'Failed to authenticate with PBX: ' . $e->getMessage()]);
                exit;
            }
        } else {
            error_log("Conexion encontrada para cliente: " . $clientId);
            
            try {
                $token = $this->decryptToken($connection->bearer_token);
                error_log("Token desencriptado correctamente");
            } catch (\Exception $e) {
                error_log("Error al desencriptar token: " . $e->getMessage());
                http_response_code(500);
                echo json_encode(['error' => 'PBX token could not be decrypted']);
                exit;
            }
        }

        try {
            $pbxService = $this->getPbxService();
            
            // Check if token is expired and refresh if needed
            error_log("Verificando si token expiro...");
            $validToken = $pbxService->getValidToken($token);
            
            // If token was refreshed, store the new one
            if ($validToken !== $token) {
                error_log("Token fue refrescado automaticamente");
                $this->storePbxConnection($clientId, $validToken);
            } else {
                error_log("Token sigue valido");
            }

            error_log("Obteniendo datos CDR con parametros: " . json_encode($params));
            $cdrResponse = $pbxService->getCdrData($validToken, $params);

            error_log("CDR obtenido exitosamente: " . count($cdrResponse['data'] ?? []) . " registros");
            error_log("JSON completo de la respuesta: " . json_encode($cdrResponse));

            http_response_code(200);
            echo json_encode([
                'success' => true,
                'data' => $cdrResponse['data'],
                'meta' => $cdrResponse['meta'],
                'total' => $cdrResponse['total'],
                'total_raw' => $cdrResponse['total_raw'] ?? null,
                'timezone' => $cdrResponse['timezone'] ?? null,
                'pagination' => $cdrResponse['pagination'] ?? null,
                'params' => $params
            ]);
            exit;
        } catch (\Exception $e) {
            error_log("Error en CDR: " . $e->getMessage());
            error_log("Stack trace: " . $e->getTraceAsString());
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
            exit;
        }
    }

    // -------------------------
    // 3) Estado de conexión PBX
    // GET /api/pbx/connection/status
    // -------------------------
    public function syncCdrCache()
    {
        header('Content-Type: application/json');
        if (!$this->failNotConfigured()) {
            exit;
        }

        $data = json_decode(file_get_contents('php://input'), true) ?: [];
        $params = [
            'start_date' => $data['initDate'] ?? null,
            'end_date' => $data['endDate'] ?? null,
            'limit' => isset($data['limit']) ? (int) $data['limit'] : 50,
            'order' => $data['order'] ?? '-calldate',
            'filters' => $data['filters'] ?? null,
            'page_delay_ms' => isset($data['page_delay_ms']) ? (int) $data['page_delay_ms'] : null,
            'max_pages' => isset($data['max_pages']) ? (int) $data['max_pages'] : null
        ];

        $clientId = 1;
        $connection = $this->getStoredPbxConnection($clientId);
        $token = null;

        if (!$connection) {
            try {
                $pbxService = $this->getPbxService();
                $authResult = $pbxService->authenticate();
                if (!$authResult || !isset($authResult['token'])) {
                    http_response_code(500);
                    echo json_encode(['error' => 'PBX authentication failed']);
                    exit;
                }
                $token = $authResult['token'];
                $expiresAt = $authResult['expires_at'] ?? null;
                $this->storePbxConnection($clientId, $token, $expiresAt);
            } catch (\Exception $e) {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to authenticate with PBX: ' . $e->getMessage()]);
                exit;
            }
        } else {
            try {
                $token = $this->decryptToken($connection->bearer_token);
            } catch (\Exception $e) {
                http_response_code(500);
                echo json_encode(['error' => 'PBX token could not be decrypted']);
                exit;
            }
        }

        try {
            $pbxService = $this->getPbxService();
            $validToken = $pbxService->getValidToken($token);
            if ($validToken !== $token) {
                $this->storePbxConnection($clientId, $validToken);
            }

            $syncResult = $pbxService->syncAggregatedCdr($validToken, $params);
            http_response_code(200);
            echo json_encode([
                'success' => true,
                'data' => $syncResult
            ]);
            exit;
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
            exit;
        }
    }

    public function connectionStatus()
    {
        header('Content-Type: application/json');
        if (!$this->failNotConfigured()) {
            exit;
        }

        $clientId = 1;

        $connection = $this->getStoredPbxConnection($clientId);

        $payload = [
            'connected' => (bool) $connection,
            'status' => $connection->status ?? null,
            'last_sync' => $connection->last_sync ?? null,
            'expires_at' => $connection->expires_at ?? null,
        ];

        // Check if token is expired
        if ($connection) {
            try {
                $token = $this->decryptToken($connection->bearer_token);
                $pbxService = $this->getPbxService();
                $payload['token_expired'] = $pbxService->isTokenExpired($token);
            } catch (\Exception $e) {
                $payload['token_error'] = 'Token validation failed';
            }
        }

        http_response_code(200);
        echo json_encode($payload);
        exit;
    }

    // -------------------------
    // 4) Refresh token
    // POST /api/pbx/refresh_token
    // -------------------------
    public function refreshToken()
    {
        header('Content-Type: application/json');
        if (!$this->failNotConfigured()) {
            exit;
        }

        $clientId = 1;

        try {
            $pbxService = $this->getPbxService();
            $authResponse = $pbxService->authenticate();

            // Store the new encrypted token
            $this->storePbxConnection($clientId, $authResponse['token'], $authResponse['expires_at']);

            http_response_code(200);
            echo json_encode([
                'success' => true,
                'token' => $authResponse['token'],
                'expires_at' => $authResponse['expires_at'],
                'message' => 'Token refreshed successfully'
            ]);
            exit;
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
            exit;
        }
    }
}
