<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use GuzzleHttp\Client as GuzzleHttpClient;

class SmartyAPIController extends BaseController
{
    public function suggestAddress()
    {
        header('Content-Type: application/json');

        try {
            $query = _get('query');
            $selected = _get('selected');

            if ($query === '') {
                echo json_encode([
                    'status' => 'success',
                    'suggestions' => [],
                    'message' => 'No query provided'
                ]);
                exit;
            }

            $authId = $_ENV['SMARTY_AUTH_ID'] ?? '';
            $authToken = $_ENV['SMARTY_AUTH_TOKEN'] ?? '';
            $license = $_ENV['SMARTY_LICENSE'] ?? '';

            if ($authId === '' || $authToken === '' || $license === '') {
                http_response_code(400);
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Missing Smarty credentials (SMARTY_AUTH_ID, SMARTY_AUTH_TOKEN, SMARTY_LICENSE)'
                ]);
                exit;
            }

            $queryParams = [
                'auth-id' => $authId,
                'auth-token' => $authToken,
                'search' => $query,
                'max_results' => 10,
                'prefer_geolocation' => 'none',
                'license' => $license,
                'source' => 'all'
            ];

            $client = new GuzzleHttpClient([
                'verify' => false,
                'curl' => [
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_ENCODING => '',
                    CURLOPT_MAXREDIRS => 10,
                    CURLOPT_TIMEOUT => 30,
                    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1
                ]
            ]);
            
            $apiResponse = $client->get("https://us-autocomplete-pro.api.smarty.com/lookup", [
                'query' => $queryParams,
                'headers' => [
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json'
                ]
            ]);
            
            $statusCode = $apiResponse->getStatusCode();
            $body = $apiResponse->getBody()->getContents();
            $suggestions = json_decode($body, true);
            
            if ($statusCode !== 200) {
                http_response_code($statusCode);
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Smarty API error',
                    'detail' => $body
                ]);
                exit;
            }

            $formattedSuggestions = array_map(function($suggestion) {
                return [
                    'label' => $suggestion['street_line'] ?? '',
                    'value' => [
                        'street_line' => $suggestion['street_line'] ?? '',
                        'secondary' => $suggestion['secondary'] ?? '',
                        'city' => $suggestion['city'] ?? '',
                        'state' => $suggestion['state'] ?? '',
                        'zipcode' => $suggestion['zipcode'] ?? '',
                        'entries' => $suggestion['entries'] ?? 0
                    ]
                ];
            }, $suggestions['suggestions'] ?? []);

            echo json_encode([
                'status' => 'success',
                'suggestions' => $formattedSuggestions,
                'message' => empty($formattedSuggestions) ? 'No suggestions found' : ''
            ]);
            exit;

        } catch (\Exception $e) {
            http_response_code(400);
            echo json_encode([
                'status' => 'error',
                'message' => 'Error suggesting address',
                'detail' => $e->getMessage()
            ]);
            exit;
        }

    }

    public function selectUnit()
    {
        header('Content-Type: application/json');

        try {
            $search = trim(_get('search'));
            $selected = trim(_get('selected'));

            if ($search === '' && $selected !== '') {
                $search = $selected;
            }
            if ($search === '') {
                http_response_code(400);
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Missing search parameter for unit lookup'
                ]);
                exit;
            }

            $authId = $_ENV['SMARTY_AUTH_ID'] ?? '';
            $authToken = $_ENV['SMARTY_AUTH_TOKEN'] ?? '';
            $license = $_ENV['SMARTY_LICENSE'] ?? '';

            if ($authId === '' || $authToken === '' || $license === '') {
                http_response_code(400);
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Missing Smarty credentials (SMARTY_AUTH_ID, SMARTY_AUTH_TOKEN, SMARTY_LICENSE)'
                ]);
                exit;
            }

            $queryParams = [
                'auth-id' => $authId,
                'auth-token' => $authToken,
                'search' => $search,
                'selected' => $selected,
                'include_only_cities' => '',
                'include_only_states' => '',
                'include_only_zip_codes' => '',
                'exclude_states' => '',
                'prefer_cities' => '',
                'prefer_states' => '',
                'prefer_zip_codes' => '',
                'source' => 'all',
                'license' => $license
            ];

            $client = new GuzzleHttpClient([
                'verify' => false,
                'curl' => [
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_ENCODING => '',
                    CURLOPT_MAXREDIRS => 10,
                    CURLOPT_TIMEOUT => 30,
                    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1
                ]
            ]);

            $apiResponse = $client->get("https://us-autocomplete-pro.api.smarty.com/lookup", [
                'query' => $queryParams,
                'headers' => [
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json'
                ]
            ]);

            $statusCode = $apiResponse->getStatusCode();
            $body = $apiResponse->getBody()->getContents();
            $suggestions = json_decode($body, true);
            
            if ($statusCode !== 200) {
                http_response_code($statusCode);
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Smarty API error',
                    'detail' => $body
                ]);
                exit;
            }

            $formattedSuggestions = array_map(function($suggestion) {
                $unitInfo = '';
                if ($suggestion['entries'] > 1) {
                    $unitInfo = ' (Multiple units available)';
                } elseif (!empty($suggestion['secondary'])) {
                    $unitInfo = ' (Unit: ' . $suggestion['secondary'] . ')';
                }

                return [
                    'label' => ($suggestion['street_line'] ?? '') . $unitInfo,
                    'value' => [
                        'street_line' => $suggestion['street_line'] ?? '',
                        'secondary' => $suggestion['secondary'] ?? '',
                        'city' => $suggestion['city'] ?? '',
                        'state' => $suggestion['state'] ?? '',
                        'zipcode' => $suggestion['zipcode'] ?? '',
                        'entries' => $suggestion['entries'] ?? 0,
                        'unit' => $suggestion['secondary'] ?? '',
                        'full_address' => ($suggestion['street_line'] ?? '') . 
                                        (!empty($suggestion['secondary']) ? ' ' . $suggestion['secondary'] : '') . 
                                        ', ' . ($suggestion['city'] ?? '') . 
                                        ', ' . ($suggestion['state'] ?? '') . 
                                        ' ' . ($suggestion['zipcode'] ?? '')
                    ]
                ];
            }, $suggestions['suggestions'] ?? []);

            echo json_encode([
                'status' => 'success',
                'suggestions' => $formattedSuggestions
            ]);
            exit;

        } catch (\Exception $e) {
            http_response_code(400);
            echo json_encode([
                'status' => 'error',
                'message' => 'Error selecting unit',
                'detail' => $e->getMessage()
            ]);
            exit;
        }
    }
}
