<?php
namespace App\Controllers;

use App\Controllers\BaseController;
use ORM;
use Twilio\Rest\Client as TwilioClient;
use SendGrid\Mail\Mail;
use SendGrid;
use PDO;

error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

/**
 * Controller para gestionar el envío de mensajes de pagos recibidos
 * (WhatsApp / SMS / Email) reutilizando la lógica del template
 * "enviar_mensajes_pagos_recibidos".
 */
class TemplateInvoicesController extends BaseController
{
    /**
     * Endpoint: procesa el envío de mensajes de pago recibido.
     * Espera JSON con: invoices_id, tipo_pago, factura_nueva.
     */
    
    public static function sendReceivedPaymentNotification($data)
    {
        $instance = new self();
        return $instance->processNotification($data);
    }

    private function processNotification(array $data): array
    {
        $invoicesId   = $data['invoices_id']   ?? null;
        $tipoPago     = $data['tipo_pago']     ?? null;
        $facturaNueva = $data['factura_nueva'] ?? null;

        if (empty($facturaNueva)) {
            return ['type' => 'error', 'message' => 'A new invoice is required'];
        }
        if (empty($tipoPago)) {
            return ['type' => 'error', 'message' => 'Payment type is required'];
        }
        if (empty($invoicesId)) {
            return ['type' => 'error', 'message' => 'Active/Inactive Template is required'];
        }

        $pdo = ORM::get_db();
        $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
        $dbName = DB_NAME;

        try {
            $currencyCode = $this->getAppConfig($pdo, 'currency_code') ?? '$';
            $urlSockets   = rtrim($this->getAppConfig($pdo, 'socket_webhook_sms') ?? '', '/');

            $invoice = $this->fetchInvoice($pdo, $dbName, $invoicesId);
            if (!$invoice) {
                return ['type' => 'error', 'message' => 'Invoices not found'];
            }

            $queryTemplateInvoices = ($facturaNueva === "true" && $invoice['rating_review_user_crm_accounts'] != 1)
                ? " AND ti.tipo_pago IN (:tipo_pago, 'resena')"
                : " AND ti.tipo_pago = :tipo_pago ";

            $templates = $this->fetchTemplates($pdo, $dbName, $queryTemplateInvoices, $tipoPago);
            if (empty($templates)) {
                return ['type' => 'error', 'message' => 'Template Invoices not found'];
            }

            $results = [];
            foreach ($templates as $tpl) {
                if ($tpl['channel'] === 'whatsapp') {
                    $results[] = $this->sendWhatsappTemplate($tpl, $invoice, $currencyCode, $urlSockets);
                } elseif ($tpl['channel'] === 'sms') {
                    $results[] = $this->sendSmsTemplate($tpl, $invoice, $currencyCode, $urlSockets);
                } elseif ($tpl['channel'] === 'email') {
                    $results[] = $this->sendEmailTemplate($tpl, $invoice, $currencyCode);
                }
            }

            return ['type' => 'success', 'message' => 'Process Completed', 'results' => $results];

        } catch (\Exception $e) {
            return ['type' => 'error', 'message' => 'Database connection failed', 'detalle' => $e->getMessage()];
        }
    }

    private function jsonError($message, array $extra = [])
    {
        return array_merge(['type' => 'error', 'message' => $message], $extra);
    }

    private function jsonSuccess($message, array $extra = [])
    {
        return array_merge(['type' => 'success', 'message' => $message], $extra);
    }

    private function getAppConfig(PDO $pdo, $key)
    {
        $stmt = $pdo->prepare("SELECT `value` FROM sys_appconfig WHERE setting = :key LIMIT 1");
        $stmt->execute([':key' => $key]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['value'] ?? null;
    }

    private function fetchInvoice(PDO $pdo, $dbName, $invoiceId)
    {
        $sql = "SELECT 
                    sys_i.*,
                    crm_a.phone  AS phone_user_crm_accounts,
                    crm_a.email  AS email_user_crm_accounts,
                    crm_a.account AS account_user_crm_accounts,
                    crm_a.rating_review AS rating_review_user_crm_accounts,
                    SUer.fullname AS user_fullname
                FROM {$dbName}.sys_invoices AS sys_i
                LEFT JOIN {$dbName}.crm_accounts AS crm_a ON crm_a.id=sys_i.userid
                LEFT JOIN {$dbName}.sys_users AS SUer ON SUer.id = sys_i.aid
                WHERE sys_i.id = :id
                LIMIT 1";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([':id' => $invoiceId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    private function fetchTemplates(PDO $pdo, $dbName, $queryTemplateInvoices, $tipoPago)
    {
        $sql = "SELECT 
                    ti.*,
                    ce.token AS config_email_token,
                    ce.email AS config_email_email,
                    ce.nombre AS config_email_nombre,
                    cwo.token AS config_whatsapp_other_token_ws,
                    cwo.phone AS config_whatsapp_other_phone_ws,
                    cwo.phone_identifier AS config_whatsapp_other_phone_identifier_ws,
                    cwo.version AS config_whatsapp_other_version_ws,
                    cwo.sms_account_sid AS config_whatsapp_other_sms_account_sid_sms,
                    cwo.sms_auth_token AS config_whatsapp_other_sms_auth_token_sms,
                    cwo.sms_phone_number AS config_whatsapp_other_sms_phone_number_sms,
                    cwo.version_sms AS config_whatsapp_other_version_sms_ws
                FROM {$dbName}.template_invoices AS ti
                LEFT JOIN {$dbName}.config_email AS ce ON ce.id=ti.config_email_id
                LEFT JOIN " . DB_CHAT_WS . ".config_whatsapp_other AS cwo ON cwo.id=ti.config_whatsapp_other_id
                WHERE 1
                    {$queryTemplateInvoices}
                    AND ti.activo_id = 1
                    AND ti.template_original IS NOT NULL AND ti.template_original != ''
                    AND ti.template_modificado IS NOT NULL AND ti.template_modificado <> ''
                    AND ti.template_data IS NOT NULL AND ti.template_data <> ''";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([':tipo_pago' => $tipoPago]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    private function sendWhatsappTemplate(array $tpl, array $invoice, $currencyCode, $urlSockets)
    {
        $phone = $invoice['phone_user_crm_accounts'] ?? '';
        if (empty($phone)) {
            return ['channel' => 'whatsapp', 'status' => 'skipped', 'reason' => 'missing phone'];
        }

        $phone = $this->normalizePhone($phone);
        $versionApi = $tpl['config_whatsapp_other_version_ws'];
        $phoneId    = $tpl['config_whatsapp_other_phone_identifier_ws'];
        $token      = $tpl['config_whatsapp_other_token_ws'];

        if (empty($versionApi) || empty($phoneId) || empty($token)) {
            return ['channel' => 'whatsapp', 'status' => 'skipped', 'reason' => 'missing whatsapp config'];
        }

        $templateMod = json_decode($tpl['template_modificado'], true);
        $templateData = json_decode($tpl['template_data'], true);

        $nuevo = [
            "messaging_product" => "whatsapp",
            "to" => $phone,
            "type" => "template",
            "template" => [
                "namespace" => $templateMod["template"]["namespace"] ?? '',
                "name"      => $templateMod["template"]["name"] ?? '',
                "language"  => [
                    "code"  => $templateMod["template"]["language"]["code"] ?? 'en',
                ],
                "components" => [],
            ],
        ];

        // Header
        foreach ($templateMod["template"]["components"] as $component) {
            if ($component['type'] == "header") {
                $params = $this->buildParamsFromTemplate($templateData, 'HEADER', $invoice, $currencyCode);
                if (!empty($params)) {
                    $nuevo["template"]["components"][] = [
                        "type" => "header",
                        "parameters" => $params,
                    ];
                }
            }

            if ($component['type'] == "body") {
                $bodyParams = $this->buildBodyParams($templateData, $invoice, $currencyCode);
                if (!empty($bodyParams)) {
                    $nuevo["template"]["components"][] = [
                        "type" => "body",
                        "parameters" => $bodyParams,
                    ];
                }
            }

            if ($component['type'] == "button") {
                $nuevo["template"]["components"][] = [
                    "type"       => "button",
                    "sub_type"   => $component['sub_type'],
                    "index"      => $component['index'],
                    "parameters" => $component['parameters'] ?? [],
                ];
            }
        }

        $url = "https://graph.facebook.com/{$versionApi}/{$phoneId}/messages";
        // Asumimos que existe la función global enviar_plantilla
        $resp = enviar_plantilla($url, $token, $nuevo);

        return ['channel' => 'whatsapp', 'status' => $resp['status'] ?? 'unknown', 'response' => $resp];
    }

    private function sendSmsTemplate(array $tpl, array $invoice, $currencyCode, $urlSockets)
    {
        $phone = $invoice['phone_user_crm_accounts'] ?? '';
        if (empty($phone)) {
            return ['channel' => 'sms', 'status' => 'skipped', 'reason' => 'missing phone'];
        }
        $phone = $this->normalizePhone($phone);

        $sid   = $tpl['config_whatsapp_other_sms_account_sid_sms'];
        $token = $tpl['config_whatsapp_other_sms_auth_token_sms'];
        $from  = $tpl['config_whatsapp_other_sms_phone_number_sms'];

        if (empty($sid) || empty($token) || empty($from)) {
            return ['channel' => 'sms', 'status' => 'skipped', 'reason' => 'missing sms config'];
        }

        $templateOriginal = json_decode($tpl['template_original'], true);
        $templateData     = json_decode($tpl['template_data'], true);
        $textBody         = $templateOriginal['plantilla'] ?? '';

        foreach ($templateData as $item) {
            $buscar = $item['buscar'] ?? '';
            $texto  = $item['texto'] ?? '';
            $campo  = $item['campoDB'] ?? '';
            $isNone = $item['isSelecNone'] ?? false;

            $value = $isNone ? $texto : $this->mapInvoiceValue($invoice, $campo, $currencyCode);
            $textBody = replacePlaceholder($textBody, $buscar, $value);
        }

        $client = new TwilioClient($sid, $token);
        $db = DB_NAME;
        $crmFolder = CARPETA_CRM;
        $statusCb = $urlSockets ? $urlSockets . "/status_v1?db={$db}&url={$crmFolder}" : null;

        $message = $client->messages->create(
            "+" . $phone,
            array_filter([
                'from' => "+" . $from,
                'body' => $textBody,
                'statusCallback' => $statusCb,
            ])
        );

        return ['channel' => 'sms', 'status' => 'sent', 'sid' => $message->sid ?? null];
    }

    private function sendEmailTemplate(array $tpl, array $invoice, $currencyCode)
    {
        $emailTo   = trim($invoice['email_user_crm_accounts'] ?? '');
        $nameTo    = trim($invoice['account_user_crm_accounts'] ?? '');

        if (empty($emailTo)) {
            return ['channel' => 'email', 'status' => 'skipped', 'reason' => 'missing email'];
        }

        $token  = $tpl['config_email_token'];
        $from   = $tpl['config_email_email'];
        $fromNm = $tpl['config_email_nombre'];

        if (empty($token) || empty($from) || empty($fromNm)) {
            return ['channel' => 'email', 'status' => 'skipped', 'reason' => 'missing email config'];
        }

        $templateOriginal = json_decode($tpl['template_original'], true);
        $templateData     = json_decode($tpl['template_data'], true);
        $htmlBody         = $templateOriginal['plantilla'] ?? '';
        $subject          = $templateOriginal['subject'] ?? 'Payment received';

        foreach ($templateData as $item) {
            $buscar = $item['buscar'] ?? '';
            $texto  = $item['texto'] ?? '';
            $campo  = $item['campoDB'] ?? '';
            $isNone = $item['isSelecNone'] ?? false;

            $value = $isNone ? $texto : $this->mapInvoiceValue($invoice, $campo, $currencyCode);
            $htmlBody = replacePlaceholder($htmlBody, $buscar, $value);
        }

        $mail = new Mail();
        $mail->setFrom($from, $fromNm);
        $mail->setSubject($subject);
        $mail->addTo($emailTo, $nameTo);
        $mail->addContent('text/html', $htmlBody);

        $sg = new SendGrid($token);
        $resp = $sg->send($mail);

        return ['channel' => 'email', 'status' => 'sent', 'code' => $resp->statusCode() ?? null];
    }

    private function buildParamsFromTemplate(array $templateData, $type, array $invoice, $currencyCode)
    {
        $params = [];
        foreach ($templateData as $item) {
            if (($item['type'] ?? '') !== $type) {
                continue;
            }
            $campo = $item['campoBD'] ?? '';
            $text  = $item['contenido'] ?? '';
            if ($campo === "0" || $campo === 0) {
                $params[] = ["type" => "text", "text" => $text];
            } else {
                $value = $this->mapInvoiceValue($invoice, $campo, $currencyCode);
                $params[] = ["type" => "text", "text" => $value];
            }
        }
        return $params;
    }

    private function buildBodyParams(array $templateData, array $invoice, $currencyCode)
    {
        $params = [];
        foreach ($templateData as $item) {
            if (($item['type'] ?? '') !== 'BODY') {
                continue;
            }
            $contenido = $item['contenido'] ?? [];
            foreach ($contenido as $c) {
                $buscar  = $c['buscar'] ?? '';
                $campoBD = $c['campoBD'] ?? '';
                $isNone  = ($c['campoBD'] ?? '') === "0" || ($c['campoBD'] ?? 0) === 0;
                $texto   = $c['texto'] ?? '';
                $value   = $isNone ? $texto : $this->mapInvoiceValue($invoice, $campoBD, $currencyCode);
                $result  = replacePlaceholder($texto, $buscar, $value);
                $params[] = ["type" => "text", "text" => $result];
            }
        }
        return $params;
    }

    private function mapInvoiceValue(array $invoice, $campo, $currencyCode)
    {
        switch ($campo) {
            case 'url_rate_review':
                return base64_encode($invoice['userid'] ?? '');
            case 'amount_last_paid':
                return $currencyCode . ' ' . createFromCurrency($invoice['amount_last_paid'] ?? 0, $currencyCode);
            case 'cn':
                return ($invoice['invoicenum'] ?? '') . ($invoice['cn'] ?? '');
            case 'userId':
                return $invoice['user_fullname'] ?? '';
            default:
                return $invoice[$campo] ?? '';
        }
    }

    private function normalizePhone($phone)
    {
        return onlyNumbers($phone);
    }

    private function onlyNumbers($texto)
    {
        // Usar preg_replace para eliminar todos los caracteres que no sean dígitos
        $textoSoloNumeros = preg_replace('/\D/', '', $texto);
        return $textoSoloNumeros;
    }

    private function replacePlaceholder($text, $placeholder, $value)
    {
        // Escapar el placeholder para usarlo en una expresión regular
        $escapedPlaceholder = preg_quote($placeholder, '/');
        
        // Crear la expresión regular para buscar el placeholder
        $regex = '/' . $escapedPlaceholder . '/';
    
        // Reemplazar el placeholder con el valor o mantener el placeholder si el valor está vacío
        $updatedText = preg_replace($regex, $value ?: $placeholder, $text);
    
        return $updatedText;
    }
}
