<?php
namespace App\Services;

use PDO;

class MenuService
{
    private PDO $db;
    private int $userId;
    private int $roleId;

    public function __construct(PDO $db, int $userId)
    {
        $this->db = $db;
        $this->userId = $userId;
        $user = $this->fetchUser();
        $this->roleId = (int)($user['roleid'] ?? -1);
    }

    public function buildMenu(): array
    {
        [$cats, $items] = $this->loadAccess();
        $categorias = $this->mapCategorias($cats);
        return $this->attachItems($categorias, $items);
    }

    private function fetchUser(): array
    {
        $stmt = $this->db->prepare('SELECT * FROM `sys_users` WHERE id = :id');
        $stmt->execute(['id' => $this->userId]);
        return $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    }

    private function loadAccess(): array
    {
        $isAdmin = in_array($this->roleId, [0, 4], true);
        $roleFilter = $isAdmin ? 'rol_admin = 1' : 'rol_empleado = 1';

        $sqlCats = "SELECT * FROM spp_menu_categoria_acceso WHERE activo = 1 AND {$roleFilter} AND acceso = 1 ORDER BY orden ASC";
        $cats = $this->db->query($sqlCats)->fetchAll(PDO::FETCH_ASSOC);

        $sqlItems = "SELECT * FROM spp_menu_item_acceso WHERE activo = 1 AND {$roleFilter} AND acceso = 1 ORDER BY orden ASC";
        $items = $this->db->query($sqlItems)->fetchAll(PDO::FETCH_ASSOC);

        return [$cats, $items];
    }

    private function mapCategorias(array $rows): array
    {
        $out = [];
        foreach ($rows as $row) {
            $out[] = [
                'id' => (int)$row['id'],
                'UUID' => $row['UUID'] ?? null,
                'UUID_ENC' => $row['UUID_ENC'] ?? null,
                'modulo_principal' => $row['nombre'] ?? null,
                'nombre' => $row['nombre'] ?? null,
                'icono' => $row['icono'] ?? null,
                'key' => $row['key'] ?? null,
                'acceso' => (int)($row['acceso'] ?? 0),
                'acceso_inhabilitado' => (int)($row['acceso_inhabilitado'] ?? 0),
                'url' => $row['url'] ?? null,
                'activo' => (int)($row['activo'] ?? 0),
                'orden' => (int)($row['orden'] ?? 0),
                'rol_admin' => (int)($row['rol_admin'] ?? 0),
                'rol_empleado' => (int)($row['rol_empleado'] ?? 0),
                'subpermisos' => []
            ];
        }
        return $out;
    }

    private function attachItems(array $categorias, array $items): array
    {
        foreach ($items as $row) {
            $catId = (int)($row['ssp_cma_id'] ?? 0);
            if (!$catId) {
                continue;
            }

            foreach ($categorias as &$cat) {
                if ($cat['id'] !== $catId) {
                    continue;
                }

                $url = $row['url'];
                if ($this->roleId === 3 && ($row['key'] ?? '') === 'mnu-8-1') {
                    $url = 'social/social-chat-employee';
                }

                $cat['subpermisos'][] = [
                    'id_modulo_principal' => $catId,
                    'modulo_principal' => $cat['nombre'],
                    'id' => (int)$row['id'],
                    'UUID' => $row['UUID'] ?? null,
                    'UUID_ENC' => $row['UUID_ENC'] ?? null,
                    'nombre' => $row['nombre'] ?? null,
                    'key' => $row['key'] ?? null,
                    'acceso' => (int)($row['acceso'] ?? 0),
                    'acceso_inhabilitado' => (int)($row['acceso_inhabilitado'] ?? 0),
                    'url' => $url,
                    'activo' => (int)($row['activo'] ?? 0),
                    'orden' => (int)($row['orden'] ?? 0),
                    'rol_admin' => (int)($row['rol_admin'] ?? 0),
                    'rol_empleado' => (int)($row['rol_empleado'] ?? 0)
                ];
                break;
            }
            unset($cat);
        }
        return $categorias;
    }
}
