<?php
namespace App\Services;

use InvalidArgumentException;
use RuntimeException;
use ORM;
use Carbon\Carbon;

class PbxService
{
    private string $baseUrl;
    private string $email;
    private string $password;

    public function __construct(array $config = [])
    {
        $this->baseUrl = $config['base_url'];
        $this->email = $config['email'];
        $this->password = $config['password'];
    }

    /**
     * Authenticate with PBX API and return bearer token
     */
    public function authenticate(): array
    {
        $ch = curl_init();
        
        $headers = [
            'accept: application/json',
            'Content-Type: application/json'
        ];

        curl_setopt_array($ch, [
            CURLOPT_URL => $this->baseUrl . '/login',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_POSTFIELDS => json_encode([
                'email' => $this->email,
                'password' => $this->password
            ]),
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            throw new RuntimeException('PBX authentication request failed: ' . $error);
        }

        $decoded = json_decode($response, true);
        if ($decoded === null) {
            throw new RuntimeException('PBX authentication response decode error (HTTP ' . $httpCode . '): ' . $response);
        }

        if ($httpCode !== 200) {
            throw new RuntimeException('PBX authentication failed (HTTP ' . $httpCode . '): ' . ($decoded['error'] ?? $response));
        }

        if (!isset($decoded['access_token'])) {
            throw new RuntimeException('PBX authentication response missing access_token');
        }

        // Calculate expiration time from expires_in (seconds)
        $expiresAt = null;
        if (isset($decoded['expires_in'])) {
            $expiresAt = date('Y-m-d H:i:s', time() + $decoded['expires_in']);
        }

        return [
            'token' => $decoded['access_token'],
            'token_type' => $decoded['token_type'] ?? 'bearer',
            'expires_in' => $decoded['expires_in'] ?? null,
            'expires_at' => $expiresAt,
            'raw_response' => $decoded
        ];
    }

    private function getPbxGroups(): array
    {
        return [
            'groupA' => [
                ['id' => 280, 'name' => 'Ninoska Chirinos'],
                ['id' => 285, 'name' => 'Michelle Cerritos'],
                ['id' => 290, 'name' => 'Cristhian Moreno'],
            ],
            'groupB' => [
                ['id' => 288, 'name' => 'Michell Paz'],
                ['id' => 273, 'name' => 'Jose Gutierrez'],
                ['id' => 296, 'name' => 'Julia Doubleday'],
            ],
            'groupC' => [
                ['id' => 287, 'name' => 'Heidy Calix'],
                ['id' => 286, 'name' => 'Emilio Pagoaga'],
                ['id' => 301, 'name' => 'Sara Rachell Orellana']
            ],
        ];
    }

    private function mapCdrByGroups(array $cdrRows): array
    {
        $groups = $this->getPbxGroups();
        $groupedStats = [];
        $nameIndex = [];

        foreach ($groups as $groupKey => $members) {
            $groupedStats[$groupKey] = [];
            foreach ($members as $member) {
                $groupedStats[$groupKey][] = [
                    'id' => $member['id'],
                    'name' => $member['name'],
                    'incoming' => 0,
                    'outgoing' => 0,
                ];
                $nameIndex[strtolower($member['name'])] = [
                    'id' => $member['id'],
                    'group' => $groupKey,
                    'index' => count($groupedStats[$groupKey]) - 1,
                ];
            }
        }

        foreach ($cdrRows as $call) {
            $direction = strtolower((string) ($call['direction'] ?? ''));
            $srcName = strtolower((string) ($call['src_name'] ?? ''));
            $dstName = strtolower((string) ($call['dst_name'] ?? ''));
            
            $dstUser = (string) ($call['dst_user'] ?? '');
            $srcUser = (string) ($call['src_user'] ?? '');

            $matched = false;
            
            $extensionToMatch = '';
            if ($direction === 'incoming' || $direction === 'inbound') {
                $extensionToMatch = $dstUser;
            } elseif ($direction === 'outgoing' || $direction === 'outbound') {
                $extensionToMatch = $srcUser;
            }
            
            if ($extensionToMatch !== '') {
                foreach ($nameIndex as $memberName => $location) {
                    if ($location['id'] == $extensionToMatch) {
                        if ($direction === 'incoming' || $direction === 'inbound') {
                            $groupedStats[$location['group']][$location['index']]['incoming']++;
                        } elseif ($direction === 'outgoing' || $direction === 'outbound') {
                            $groupedStats[$location['group']][$location['index']]['outgoing']++;
                        }
                        $matched = true;
                        break;
                    }
                }
            }

            if (!$matched) {
                $matchTarget = '';
                if ($direction === 'incoming' || $direction === 'inbound') {
                    $matchTarget = $dstName !== '' ? $dstName : $srcName;
                } elseif ($direction === 'outgoing' || $direction === 'outbound') {
                    $matchTarget = $srcName !== '' ? $srcName : $dstName;
                } else {
                    $matchTarget = $srcName !== '' ? $srcName : $dstName;
                }

                if ($matchTarget !== '') {
                    foreach ($nameIndex as $memberName => $location) {
                        if (strpos($matchTarget, $memberName) !== false) {
                            if ($direction === 'incoming' || $direction === 'inbound') {
                                $groupedStats[$location['group']][$location['index']]['incoming']++;
                            } elseif ($direction === 'outgoing' || $direction === 'outbound') {
                                $groupedStats[$location['group']][$location['index']]['outgoing']++;
                            }
                            $matched = true;
                            break;
                        }
                    }
                }
            }

        }

        return $groupedStats;
    }

    private function countGroupedCalls(array $groupedData): int
    {
        $total = 0;
        foreach ($groupedData as $members) {
            if (!is_array($members)) {
                continue;
            }
            foreach ($members as $member) {
                $total += (int) ($member['incoming'] ?? 0);
                $total += (int) ($member['outgoing'] ?? 0);
            }
        }
        return $total;
    }

    private function getMemberIndex(): array
    {
        $index = [];
        foreach ($this->getPbxGroups() as $groupKey => $members) {
            foreach ($members as $member) {
                $index[strtolower($member['name'])] = [
                    'group' => $groupKey,
                    'id' => $member['id'],
                    'name' => $member['name'],
                ];
            }
        }
        return $index;
    }

    private function getDateFromCall(array $call, ?string $timezone = null): string
    {
        $raw = $call['create_time'] ?? $call['update_time'] ?? $call['calldate'] ?? $call['call_date'] ?? $call['date'] ?? $call['start_time'] ?? null;
        $tz = new \DateTimeZone($timezone ?: date_default_timezone_get());

        if (is_numeric($raw)) {
            $dt = new \DateTimeImmutable('@' . (string) $raw);
            return $dt->setTimezone($tz)->format('Y-m-d');
        }

        if ($raw) {
            try {
                $dt = new \DateTimeImmutable((string) $raw, $tz);
                return $dt->format('Y-m-d');
            } catch (\Throwable $e) {
            }
        }

        return (new \DateTimeImmutable('now', $tz))->format('Y-m-d');
    }

    private function resolveDateRange(array $params): array
    {
        $initDate = $params['initDate'] ?? $params['start_date'] ?? null;
        $endDate = $params['endDate'] ?? $params['end_date'] ?? null;

        $filters = $params['filters'] ?? null;
        if (is_string($filters)) {
            $decoded = json_decode($filters, true);
            if (!is_array($decoded)) {
                $decoded = json_decode(stripslashes($filters), true);
            }
            if (is_array($decoded)) {
                $filters = $decoded;
            }
        }

        if (is_array($filters)) {
            $initDate = $initDate ?? ($filters['date']['initDate'] ?? null);
            $endDate = $endDate ?? ($filters['date']['endDate'] ?? null);
            $initDate = $initDate ?? ($filters['date']['start_date'] ?? null);
            $endDate = $endDate ?? ($filters['date']['end_date'] ?? null);
        }

        if (!$initDate) {
            $initDate = date('Y-m-d');
        }
        if (!$endDate) {
            $endDate = $initDate;
        }

        return [
            'start_date' => $initDate,
            'end_date' => $endDate
        ];
    }

    public function getDateRangeFromParams(array $params): array
    {
        return $this->resolveDateRange($params);
    }

    private function matchCallToMember(array $call, array $nameIndex): ?array
    {
        $direction = strtolower((string) ($call['direction'] ?? ''));
        $srcName = strtolower((string) ($call['src_name'] ?? ''));
        $dstName = strtolower((string) ($call['dst_name'] ?? ''));
        
        $dstUser = (string) ($call['dst_user'] ?? '');
        $srcUser = (string) ($call['src_user'] ?? '');

        $extensionToMatch = '';
        if ($direction === 'incoming' || $direction === 'inbound') {
            $extensionToMatch = $dstUser;
        } elseif ($direction === 'outgoing' || $direction === 'outbound') {
            $extensionToMatch = $srcUser;
        }
        
        if ($extensionToMatch !== '') {
            foreach ($nameIndex as $memberName => $member) {
                if ($member['id'] == $extensionToMatch) {
                    $member['direction'] = $direction;
                    return $member;
                }
            }
        }

        $matchTarget = '';
        if ($direction === 'incoming' || $direction === 'inbound') {
            $matchTarget = $dstName !== '' ? $dstName : $srcName;
        } elseif ($direction === 'outgoing' || $direction === 'outbound') {
            $matchTarget = $srcName !== '' ? $srcName : $dstName;
        } else {
            $matchTarget = $srcName !== '' ? $srcName : $dstName;
        }

        if ($matchTarget !== '') {
            foreach ($nameIndex as $memberName => $member) {
                if (strpos($matchTarget, $memberName) !== false) {
                    $member['direction'] = $direction;
                    return $member;
                }
            }
        }

        return null;
    }

    private function buildAggregates(array $cdrRows, ?string $timezone = null): array
    {
        $nameIndex = $this->getMemberIndex();
        $aggregates = [];

        foreach ($cdrRows as $call) {
            $match = $this->matchCallToMember($call, $nameIndex);
            if (!$match) {
                continue;
            }

            $cdrDate = $this->getDateFromCall($call, $timezone);

            if (!isset($aggregates[$cdrDate])) {
                $aggregates[$cdrDate] = [
                    'rows' => []
                ];
            }

            $rowKey = $match['group'] . '|' . $match['id'];
            if (!isset($aggregates[$cdrDate]['rows'][$rowKey])) {
                $aggregates[$cdrDate]['rows'][$rowKey] = [
                    'group_key' => $match['group'],
                    'member_id' => $match['id'],
                    'member_name' => $match['name'],
                    'incoming' => 0,
                    'outgoing' => 0
                ];
            }

            if ($match['direction'] === 'incoming' || $match['direction'] === 'inbound') {
                $aggregates[$cdrDate]['rows'][$rowKey]['incoming']++;
            } elseif ($match['direction'] === 'outgoing' || $match['direction'] === 'outbound') {
                $aggregates[$cdrDate]['rows'][$rowKey]['outgoing']++;
            }
        }

        return $aggregates;
    }

    private function countAggregates(array $aggregates): int
    {
        $count = 0;
        foreach ($aggregates as $weekData) {
            $count += count($weekData['rows'] ?? []);
        }
        return $count;
    }

    private function getSyncState(): ?string
    {
        if (!function_exists('db_table_exist') || !db_table_exist('pbx_cdr_sync_state')) {
            return null;
        }

        $row = ORM::for_table('pbx_cdr_sync_state')
            ->where('id', 1)
            ->find_one();

        return $row ? $row->last_sync_at : null;
    }

    public function hasSyncState(): bool
    {
        return $this->getSyncState() !== null;
    }

    public function getSyncDate(): ?string
    {
        $lastSync = $this->getSyncState();
        if (!$lastSync) {
            return null;
        }
        $date = date('Y-m-d', strtotime($lastSync));
        return $date ?: null;
    }

    private function updateSyncState(string $timestamp): void
    {
        if (!function_exists('db_table_exist') || !db_table_exist('pbx_cdr_sync_state')) {
            return;
        }

        $pdo = ORM::get_db();
        $sql = "INSERT INTO pbx_cdr_sync_state (id, last_sync_at, updated_at)
                VALUES (1, :last_sync_at, NOW())
                ON DUPLICATE KEY UPDATE
                last_sync_at = :last_sync_at,
                updated_at = NOW()";
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['last_sync_at' => $timestamp]);
    }

    private function buildDefaultSyncFilters(string $startDate, string $endDate): string
    {
        $filters = [
            'direction' => [
                'incoming' => true,
                'outgoing' => true,
                'internal' => false
            ],
            'callStatus' => [
                'answered' => true,
                'notAnswered' => false
            ],
            'date' => [
                'initDate' => $startDate,
                'endDate' => $endDate
            ]
        ];

        return json_encode($filters);
    }

    private function clearAggregatesForRange(string $startDate, string $endDate): void
    {
        if (!function_exists('db_table_exist') || !db_table_exist('pbx_cdr_cache_agg_daily')) {
            return;
        }

        ORM::for_table('pbx_cdr_cache_agg_daily')
            ->where_gte('cdr_date', $startDate)
            ->where_lte('cdr_date', $endDate)
            ->delete_many();
    }

    private function saveAggregates(array $aggregates): void
    {
        if (!function_exists('db_table_exist') || !db_table_exist('pbx_cdr_cache_agg_daily')) {
            return;
        }

        $pdo = ORM::get_db();
        $sql = "INSERT INTO pbx_cdr_cache_agg_daily
                (cdr_date, group_key, member_id, member_name, incoming, outgoing, updated_at)
                VALUES
                (:cdr_date, :group_key, :member_id, :member_name, :incoming, :outgoing, NOW())
                ON DUPLICATE KEY UPDATE
                member_name = VALUES(member_name),
                incoming = VALUES(incoming),
                outgoing = VALUES(outgoing),
                updated_at = NOW()";

        $stmt = $pdo->prepare($sql);
        foreach ($aggregates as $cdrDate => $dateData) {
            foreach ($dateData['rows'] as $row) {
                $stmt->execute([
                    'cdr_date' => $cdrDate,
                    'group_key' => $row['group_key'],
                    'member_id' => $row['member_id'],
                    'member_name' => $row['member_name'],
                    'incoming' => $row['incoming'],
                    'outgoing' => $row['outgoing']
                ]);
            }
        }
    }

    private function isCacheAvailable(): bool
    {
        return function_exists('db_table_exist') && db_table_exist('pbx_cdr_cache');
    }

    private function getCdrCacheKey(array $params): string
    {
        $normalized = $this->normalizeCacheParams($params);
        return hash('sha256', json_encode($normalized));
    }

    private function normalizeCacheParams(array $params): array
    {
        if (isset($params['filters']) && is_string($params['filters'])) {
            $decoded = json_decode($params['filters'], true);
            if (is_array($decoded)) {
                $params['filters'] = $decoded;
            }
        }

        return $this->sortRecursive($params);
    }

    private function sortRecursive($value)
    {
        if (is_array($value)) {
            foreach ($value as $key => $item) {
                $value[$key] = $this->sortRecursive($item);
            }
            if ($this->isAssocArray($value)) {
                ksort($value);
            } else {
                sort($value);
            }
        }

        return $value;
    }

    private function isAssocArray(array $value): bool
    {
        return array_keys($value) !== range(0, count($value) - 1);
    }

    private function getCachedCdr(string $cacheKey): ?array
    {
        if (!$this->isCacheAvailable()) {
            return null;
        }

        $row = ORM::for_table('pbx_cdr_cache')
            ->where('cache_key', $cacheKey)
            ->where_gte('expires_at', date('Y-m-d H:i:s'))
            ->order_by_desc('id')
            ->find_one();

        if (!$row) {
            return null;
        }

        $data = json_decode($row->data_json ?? '', true);
        $meta = $row->meta_json ? json_decode($row->meta_json, true) : null;

        return [
            'data' => is_array($data) ? $data : [],
            'meta' => $meta,
            'total' => $row->total ?? null,
            'timezone' => $row->timezone ?? null,
        ];
    }

    private function saveCdrCache(
        string $cacheKey,
        array $params,
        array $groupedData,
        $meta,
        $total,
        $timezone,
        int $ttlSeconds
    ): void {
        if (!$this->isCacheAvailable()) {
            return;
        }

        $pdo = ORM::get_db();
        $now = date('Y-m-d H:i:s');
        $expiresAt = date('Y-m-d H:i:s', time() + max(0, $ttlSeconds));

        $sql = "INSERT INTO pbx_cdr_cache
                (cache_key, params_json, data_json, meta_json, total, timezone, fetched_at, expires_at)
                VALUES
                (:cache_key, :params_json, :data_json, :meta_json, :total, :timezone, :fetched_at, :expires_at)
                ON DUPLICATE KEY UPDATE
                params_json = :params_json,
                data_json = :data_json,
                meta_json = :meta_json,
                total = :total,
                timezone = :timezone,
                fetched_at = :fetched_at,
                expires_at = :expires_at";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            'cache_key' => $cacheKey,
            'params_json' => json_encode($params),
            'data_json' => json_encode($groupedData),
            'meta_json' => $meta ? json_encode($meta) : null,
            'total' => $total,
            'timezone' => $timezone,
            'fetched_at' => $now,
            'expires_at' => $expiresAt,
        ]);
    }

    private function requestCdrPage(string $bearerToken, array $queryParams): array
    {
        $ch = curl_init();
        $queryString = http_build_query($queryParams);

        curl_setopt_array($ch, [
            CURLOPT_URL => $this->baseUrl . '/pbx/loadCdr?' . $queryString,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPGET => true,
            CURLOPT_HTTPHEADER => [
                'accept: application/json',
                'Authorization: Bearer ' . $bearerToken
            ],
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            throw new RuntimeException('PBX CDR request failed: ' . $error);
        }

        $decoded = json_decode($response, true);
        if ($decoded === null) {
            throw new RuntimeException('PBX CDR response decode error (HTTP ' . $httpCode . '): ' . $response);
        }

        if ($httpCode !== 200) {
            throw new RuntimeException('PBX CDR request failed (HTTP ' . $httpCode . '): ' . ($decoded['error'] ?? $response));
        }

        return $decoded;
    }

    private function fetchAllCdr(
        string $bearerToken,
        array $queryParams,
        int $pageDelayMs = 150,
        int $maxPages = 50
    ): array {
        $allData = [];
        $meta = null;
        $timezone = null;
        $total = null;

        $limit = (int) ($queryParams['limit'] ?? 50);
        $page = (int) ($queryParams['page'] ?? 1);
        $pagesFetched = 0;

        while (true) {
            $queryParams['page'] = $page;
            $decoded = $this->requestCdrPage($bearerToken, $queryParams);
            $pageData = $decoded['data'] ?? [];
            $allData = array_merge($allData, $pageData);
            $pagesFetched++;

            if (isset($decoded['meta'])) {
                $meta = $decoded['meta'];
                $timezone = $decoded['meta']['timezone'] ?? $timezone;
                $total = isset($decoded['meta']['total']) ? (int) $decoded['meta']['total'] : $total;
                if (isset($decoded['meta']['limit'])) {
                    $limit = (int) $decoded['meta']['limit'];
                }
            }

            if ($total !== null && count($allData) >= $total) {
                break;
            }

            if (count($pageData) === 0) {
                break;
            }

            if ($maxPages > 0 && $pagesFetched >= $maxPages) {
                break;
            }

            $page++;
            if ($pageDelayMs > 0) {
                usleep($pageDelayMs * 1000);
            }
        }

        return [
            'data' => $allData,
            'meta' => $meta,
            'total' => $total,
            'timezone' => $timezone,
            'page' => $page,
            'limit' => $limit,
            'pages_fetched' => $pagesFetched,
            'raw_response' => $decoded ?? null
        ];
    }
    /**
     * Get CDR data from PBX API
     */
    public function getCdrData(string $bearerToken, array $params = []): array
    {
        $cacheTtl = isset($params['cache_ttl']) ? (int) $params['cache_ttl'] : 300;
        $fetchAll = array_key_exists('fetch_all', $params) ? (bool) $params['fetch_all'] : true;
        $pageDelayMs = isset($params['page_delay_ms']) ? (int) $params['page_delay_ms'] : 150;
        $maxPages = isset($params['max_pages']) ? (int) $params['max_pages'] : 50;
        $bypassCache = !empty($params['bypass_cache']);
        unset($params['cache_ttl'], $params['fetch_all'], $params['page_delay_ms'], $params['max_pages'], $params['bypass_cache']);

        $defaultParams = [
            'company' => 20152,
            'order' => '-calldate',
            'page' => 1,
            'limit' => 50,
            'month' => date('Ym'),
            'filters' => ''
        ];

        $queryParams = array_merge($defaultParams, $params);
        if ($fetchAll) {
            $queryParams['page'] = 1;
        }

        $cacheKey = $this->getCdrCacheKey($queryParams);
        if (!$bypassCache) {
            $cached = $this->getCachedCdr($cacheKey);
            if ($cached) {
                $cachedRawTotal = isset($cached['meta']['total']) ? (int) $cached['meta']['total'] : null;
                return [
                    'data' => $cached['data'],
                    'meta' => $cached['meta'],
                    'total' => $cached['total'],
                    'total_raw' => $cachedRawTotal,
                    'timezone' => $cached['timezone'],
                    'pagination' => null,
                    'raw_response' => null
                ];
            }
        }

        if ($fetchAll) {
            $response = $this->fetchAllCdr($bearerToken, $queryParams, $pageDelayMs, $maxPages);
        } else {
            $decoded = $this->requestCdrPage($bearerToken, $queryParams);
            $response = [
                'data' => $decoded['data'] ?? [],
                'meta' => $decoded['meta'] ?? null,
                'total' => $decoded['meta']['total'] ?? null,
                'timezone' => $decoded['meta']['timezone'] ?? null,
                'page' => $queryParams['page'],
                'limit' => $queryParams['limit'],
                'pages_fetched' => 1,
                'raw_response' => $decoded
            ];
        }

        $groupedData = $this->mapCdrByGroups($response['data'] ?? []);
        $totalCalls = $this->countGroupedCalls($groupedData);
        $this->saveCdrCache(
            $cacheKey,
            $queryParams,
            $groupedData,
            $response['meta'] ?? null,
            $totalCalls,
            $response['timezone'] ?? null,
            $cacheTtl
        );

        return [
            'data' => $groupedData,
            'meta' => $response['meta'] ?? null,
            'total' => $totalCalls,
            'total_raw' => $response['meta']['total'] ?? null,
            'timezone' => $response['timezone'] ?? null,
            'pagination' => [
                'page' => $response['page'] ?? null,
                'limit' => $response['limit'] ?? null,
                'pages_fetched' => $response['pages_fetched'] ?? null,
            ],
            'raw_response' => $response['raw_response'] ?? null
        ];
    }

    public function syncAggregatedCdr(string $bearerToken, array $params = []): array
    {
        $lastSync = $this->getSyncState();
        $now = time();

        $startDate = $params['start_date'] ?? null;
        $endDate = $params['end_date'] ?? null;

        if (!$startDate || !$endDate) {
            $nowCarbon = Carbon::now();
            $weekStart = $nowCarbon->copy()->startOfWeek(Carbon::MONDAY);
            $weekEnd = $nowCarbon->copy()->startOfWeek(Carbon::MONDAY)->addDays(5);

            if (!$startDate) {
                $startDate = $weekStart->toDateString();
            }
            if (!$endDate) {
                $endDate = $weekEnd->toDateString();
            }
        }

        $filters = $params['filters'] ?? null;
        if (is_array($filters)) {
            $filters = json_encode($filters);
        }
        if (!$filters) {
            $filters = $this->buildDefaultSyncFilters($startDate, $endDate);
        }

        $range = $this->resolveDateRange(['filters' => $filters]);
        $startDate = $range['start_date'];
        $endDate = $range['end_date'];

        $queryParams = [
            'order' => $params['order'] ?? '-calldate',
            'page' => 1,
            'limit' => isset($params['limit']) ? (int) $params['limit'] : 50,
            'month' => date('Ym', $now),
            'filters' => $filters
        ];

        $pageDelayMs = isset($params['page_delay_ms']) ? (int) $params['page_delay_ms'] : 150;
        $maxPages = isset($params['max_pages']) ? (int) $params['max_pages'] : 50;

        $response = $this->fetchAllCdr($bearerToken, $queryParams, $pageDelayMs, $maxPages);
        $timezone = $response['meta']['timezone'] ?? null;
        $aggregates = $this->buildAggregates($response['data'] ?? [], $timezone);
        $this->clearAggregatesForRange($startDate, $endDate);
        $this->saveAggregates($aggregates);
        $this->updateSyncState(date('Y-m-d H:i:s', $now));

        return [
            'synced' => true,
            'from' => $startDate,
            'to' => $endDate,
            'calls' => count($response['data'] ?? []),
            'total_raw' => $response['total'] ?? null,
            'pages_fetched' => $response['pages_fetched'] ?? null,
            'aggregated_rows' => $this->countAggregates($aggregates)
        ];
    }

    public function getAggregatedCdr(array $params = []): array
    {
        $range = $this->resolveDateRange($params);
        $startDate = $range['start_date'];
        $endDate = $range['end_date'];

        $rows = [];
        if (function_exists('db_table_exist') && db_table_exist('pbx_cdr_cache_agg_daily')) {
            $rows = ORM::for_table('pbx_cdr_cache_agg_daily')
                ->where_gte('cdr_date', $startDate)
                ->where_lte('cdr_date', $endDate)
                ->find_many();
        }

        $groupedData = ['groupA' => [], 'groupB' => [], 'groupC' => []];
        $memberIndex = [];
        foreach ($this->getPbxGroups() as $groupKey => $members) {
            foreach ($members as $member) {
                $groupedData[$groupKey][] = [
                    'id' => $member['id'],
                    'name' => $member['name'],
                    'incoming' => 0,
                    'outgoing' => 0
                ];
                $memberIndex[$groupKey . '|' . $member['id']] = count($groupedData[$groupKey]) - 1;
            }
        }

        foreach ($rows as $row) {
            $groupKey = $row->group_key;
            $memberId = $row->member_id;
            $indexKey = $groupKey . '|' . $memberId;
            if (isset($memberIndex[$indexKey])) {
                $idx = $memberIndex[$indexKey];
                $groupedData[$groupKey][$idx]['incoming'] += (int) $row->incoming;
                $groupedData[$groupKey][$idx]['outgoing'] += (int) $row->outgoing;
            }
        }

        $totalCalls = $this->countGroupedCalls($groupedData);

        return [
            'data' => $groupedData,
            'meta' => [
                'start_date' => $startDate,
                'end_date' => $endDate
            ],
            'total' => $totalCalls,
            'total_raw' => null,
            'timezone' => null,
            'pagination' => null,
            'raw_response' => null
        ];
    }

    /**
     * Validate token format (basic JWT validation)
     */
    public function validateTokenFormat(string $token): bool
    {
        $parts = explode('.', $token);
        return count($parts) === 3;
    }

    /**
     * Extract token expiration from JWT payload
     */
    public function getTokenExpiration(string $token): ?int
    {
        if (!$this->validateTokenFormat($token)) {
            return null;
        }

        $parts = explode('.', $token);
        $payload = json_decode(base64_decode($parts[1]), true);
        
        return $payload['exp'] ?? null;
    }

    /**
     * Check if token is expired
     */
    public function isTokenExpired(string $token): bool
    {
        $expiration = $this->getTokenExpiration($token);
        if ($expiration === null) {
            return true; // Assume expired if we can't validate
        }

        return $expiration <= time();
    }

    /**
     * Refresh token if needed
     */
    public function getValidToken(?string $storedToken = null): string
    {
        if ($storedToken && !$this->isTokenExpired($storedToken)) {
            return $storedToken;
        }

        $authResponse = $this->authenticate();
        return $authResponse['token'];
    }
}
