{extends file="$layouts_client"}

{block name="head"}
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
{/block}


{block name="content"}
<div id="plaid-app" v-cloak>

    <div v-if="!connected" class="finance-hero">
        <div class="finance-hero-intro">Manage your connected financial accounts and data sources securely.</div>
        <div class="finance-hero-card">
            <div class="finance-hero-icon">
                <svg xmlns="http://www.w3.org/2000/svg" width="800px" height="800px" viewBox="0 0 24 24" fill="none">
                    <path d="M8 6.75C5.10051 6.75 2.75 9.10051 2.75 12C2.75 14.8995 5.10051 17.25 8 17.25H9C9.41421 17.25 9.75 17.5858 9.75 18C9.75 18.4142 9.41421 18.75 9 18.75H8C4.27208 18.75 1.25 15.7279 1.25 12C1.25 8.27208 4.27208 5.25 8 5.25H9C9.41421 5.25 9.75 5.58579 9.75 6C9.75 6.41421 9.41421 6.75 9 6.75H8Z" fill="#1f6ae6"/>
                    <path d="M8.24991 11.9999C8.24991 11.5857 8.58569 11.2499 8.99991 11.2499H14.9999C15.4141 11.2499 15.7499 11.5857 15.7499 11.9999C15.7499 12.4142 15.4141 12.7499 14.9999 12.7499H8.99991C8.58569 12.7499 8.24991 12.4142 8.24991 11.9999Z" fill="#1f6ae6"/>
                    <path d="M15 5.25C14.5858 5.25 14.25 5.58579 14.25 6C14.25 6.41421 14.5858 6.75 15 6.75H16C18.8995 6.75 21.25 9.10051 21.25 12C21.25 14.8995 18.8995 17.25 16 17.25H15C14.5858 17.25 14.25 17.5858 14.25 18C14.25 18.4142 14.5858 18.75 15 18.75H16C19.7279 18.75 22.75 15.7279 22.75 12C22.75 8.27208 19.7279 5.25 16 5.25H15Z" fill="#1f6ae6"/>
                </svg>
                <span class="checkmark">
                    <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round">
                        <polyline points="20 6 9 17 4 12"></polyline>
                    </svg>
                </span>
            </div>
            <div class="finance-hero-title">Link your bank account to start tracking</div>
            <p class="finance-hero-copy">
                Connect your primary business account to automatically sync transactions. We use Plaid to securely access your financial data without ever storing your login credentials.
            </p>

            <div class="plaid-connection-act">
                    
                <button type="button" class="plaid-primary-btn" @click="connect" :disabled="loading">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M21 12H3"></path><path d="M15 18l6-6-6-6"></path>
                    </svg>
                    {{ loading ? 'Connecting...' : 'Connect with Plaid' }}
                </button>
                <div class="plaid-secured">
                    <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <rect x="3" y="11" width="18" height="11" rx="2" ry="2"></rect>
                        <path d="M7 11V7a5 5 0 0 1 10 0v4"></path>
                    </svg>
                    Secured by Plaid
                </div>
            
            </div>

            <div class="finance-feature-grid">
                <div class="finance-feature">
                    <span class="icon">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"></path>
                        </svg>
                    </span>
                    <div>
                        <h4>Bank-grade Security</h4>
                        <p>256-bit encryption standard</p>
                    </div>
                </div>
                <div class="finance-feature">
                    <span class="icon">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="12" cy="12" r="10"></circle>
                            <circle cx="12" cy="12" r="3"></circle>
                        </svg>
                    </span>
                    <div>
                        <h4>Read-only Access</h4>
                        <p>We cannot move your money</p>
                    </div>
                </div>
                <div class="finance-feature">
                    <span class="icon">
                        <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" fill="currentColor" version="1.1" width="18" height="18" viewBox="0 0 538.842 538.842" xml:space="preserve">
                            <g><g><polygon points="392.326,200.43 325.083,200.43 423.691,0 280.813,0 133.933,298.54 210.513,298.54 115.151,538.842   "/></g></g>
                        </svg>
                    </span>
                    <div>
                        <h4>Real-time Updates</h4>
                        <p>Syncs every few hours</p>
                    </div>
                </div>
            </div>
        </div>

        
        <div class="finance-footnote">
            <p>Supports 11,000+ financial institutions including Chase, Wells Fargo, and Bank of America.<br>
            By clicking "Connect with Plaid", you agree to our <a href="#" tabindex="-1">Terms of Service</a> and <a href="#" tabindex="-1">Privacy Policy</a>.</p>
        </div>
    </div>

    <div v-else class="panel">
    
        <div class="panel-hdr">
            <h2>Finance</h2>
            <div class="panel-toolbar">
                <div class="btn-group">
                    <button @click="connect" class="btn btn-primary">
                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 20 20" fill="currentColor">
                            <path d="M5 10.0002H15.0006M10.0002 5V15.0006" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                        </svg>
                        Add new bank
                    </button>
                </div>
            </div>
        </div>

        <div class="panel-container">

            <div class="panel-content">

                <div class="finance-connected">
                    <div class="finance-connected-bar">
                        <div class="finance-connected-info">
                            <div class="connected-chip">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="20 6 9 17 4 12"></polyline>
                                </svg>
                                Connected
                            </div>
                            <div>{{ institution || 'Linked bank account' }}</div>
                        </div>
                        <div class="finance-actions">
                            <input type="search" v-model="search" class="form-control input-search-table finance-search" placeholder="Search transactions">
                            <select class="form-control finance-search" v-model.number="selectedConnection" @change="handleConnectionChange" style="max-width: 240px;">
                                <option disabled :value="null">Select bank account</option>
                                <option v-for="conn in connections" :key="conn.id" :value="conn.id">
                                    {{ conn.institution_name || ('Account ' + conn.id) }}
                                </option>
                            </select>

                            <div class="datepicker-cta">
                                <button
                                    type="button"
                                    class="date-display"
                                    aria-label="Open date range picker"
                                    style="min-width:245px;"
                                    ref="dateDisplayBtn"
                                    @click="openDatepicker"
                                >
                                    <span class="icon" aria-hidden="true">
                                        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.6" stroke="currentColor">
                                            <path stroke-linecap="round" stroke-linejoin="round" d="M8 7V4m8 3V4m-9 7h10M5 20h14a1 1 0 001-1V7a1 1 0 00-1-1H5a1 1 0 00-1 1v12a1 1 0 001 1z" />
                                        </svg>
                                    </span>
                                    <span class="date-display-text">{{ rangeLabel }}</span>
                                </button>
                                <input type="text" id="dashboard-date-range" class="date-range-input" aria-hidden="true" ref="dateRangeInput" />
                            </div>

                            <button type="button" class="button" @click="getTransactions" :disabled="loading">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="23 4 23 10 17 10"></polyline>
                                    <polyline points="1 20 1 14 7 14"></polyline>
                                    <path d="M3.51 9a9 9 0 0 1 14.85-3.36L23 10"></path>
                                    <path d="M20.49 15a9 9 0 0 1-14.85 3.36L1 14"></path>
                                </svg>
                            </button>
                        </div>
                    </div>

                    <div class="table-container tc-table mt-3">
                        <div v-if="loading" class="skeleton-table">
                            <div class="skeleton-row" v-for="n in 6" :key="n">
                                <div class="skeleton-cell" style="width: 120px;"></div>
                                <div class="skeleton-cell" style="width: 60%;"></div>
                                <div class="skeleton-cell" style="width: 40%;"></div>
                                <div class="skeleton-cell" style="width: 50%;"></div>
                                <div class="skeleton-cell" style="width: 80px;"></div>
                            </div>
                        </div>
                        <table v-else id="transactions">
                            <thead>
                                <tr>
                                    <th @click="toggleSort('date')" role="button" width="120">
                                        Date
                                        <svg class="sort-icon" :class="sortKey === 'date' ? (sortDir === 'asc' ? 'is-asc' : 'is-desc') : 'is-neutral'" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
                                            <path d="M12 16l-6-6h12z"></path>
                                        </svg>
                                    </th>
                                    <th @click="toggleSort('name')" role="button">
                                        Description
                                        <svg class="sort-icon" :class="sortKey === 'name' ? (sortDir === 'asc' ? 'is-asc' : 'is-desc') : 'is-neutral'" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
                                            <path d="M12 16l-6-6h12z"></path>
                                        </svg>
                                    </th>
                                    <th class="text-start" @click="toggleSort('amount')" role="button">
                                        Amount
                                        <svg class="sort-icon" :class="sortKey === 'amount' ? (sortDir === 'asc' ? 'is-asc' : 'is-desc') : 'is-neutral'" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
                                            <path d="M12 16l-6-6h12z"></path>
                                        </svg>
                                    </th>
                                    <th>Account Name</th>
                                    <th width="150">Account Type</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr v-if="!paged.length && !loading">
                                    <td colspan="5">No transactions found.</td>
                                </tr>
                                <tr v-for="tx in paged" :key="tx.transaction_id || (tx.name + tx.date)">
                                    <td data-label="Date">{{ tx.date }}</td>
                                    <td data-label="Description">{{ tx.name }}</td>
                                    <td data-label="Amount" class="text-start" :class="tx.amount < 0 ? 'text-positive' : 'text-negative'">
                                        {{ (tx.amount || 0) * -1 }} {{ tx.iso_currency_code }}
                                    </td>
                                    <td data-label="Account Name">{{ accountName(tx.account_id) }}</td>
                                    <td data-label="Account Type">{{ accountSubtype(tx.account_id) }}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>

                    <div class="table-footer mt-2" v-if="totalTransactions">
                        <div>Showing {{ pageStart + 1 }} to {{ Math.min(pageStart + (transactions.length || 0), totalTransactions) }} of {{ totalTransactions }} entries</div>
                        <div class="d-flex align-items-center gap-2">
                            <select v-model.number="pageSize" class="form-control select-per-page">
                                <option :value="10">10 per page</option>
                                <option :value="25">25 per page</option>
                                <option :value="50">50 per page</option>
                            </select>
                            <div class="table-paging">
                                <button class="button" @click="goToPage(currentPage - 1)" :disabled="currentPage === 1"><</button>
                                <template v-for="item in pageNumbers" :key="item.key">
                                    <button
                                        v-if="item.type === 'page'"
                                        class="button"
                                        :class="item.value === currentPage ? 'button-active-page' : ''"
                                        @click="goToPage(item.value)"
                                    >
                                        {{ item.value }}
                                    </button>
                                    <button
                                        v-else
                                        class="button button-ellipsis"
                                        type="button"
                                        disabled
                                        aria-hidden="true"
                                    >
                                        ...
                                    </button>
                                </template>
                                <button class="button" @click="goToPage(currentPage + 1)" :disabled="currentPage >= totalPages">></button>
                            </div>
                        </div>
                    </div>
                </div>

            </div>

        </div>

    </div>
</div>
{/block}

{block name="script"}
<script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
<script>
const cid = '{$user->id|escape:"javascript"}';
</script>
<script>
const { createApp, ref, computed, onMounted, watch, nextTick } = Vue;

createApp({
    setup() {
        const loading = ref(false);
        const connected = ref(false);
        const institution = ref(null);
        const connections = ref([]);
        const selectedConnection = ref(null);
        const transactions = ref([]);
        const accounts = ref([]);
        const search = ref('');
        const dateRangeInput = ref(null);
        const dateDisplayBtn = ref(null);
        const rangeLabel = ref('Select range');
        const currentRange = ref('weekly');
        const currentStart = ref(null);
        const currentEnd = ref(null);
        let rangePicker = null;
        const sortKey = ref(null);
        const sortDir = ref('asc');
        const pageSize = ref(10);
        const currentPage = ref(1);
        const totalTransactions = ref(0);


        const pageNumbers = computed(() => {
            const total = totalPages.value;
            const current = currentPage.value;
            const pages = [];

            const addPage = (page) => pages.push({ldelim} type: 'page', value: page, key: 'p-' + page {rdelim});
            const addEllipsis = (id) => pages.push({ldelim} type: 'ellipsis', value: id, key: 'e-' + id + '-' + total + '-' + current {rdelim});

            if (total <= 7) {
                for (let i = 1; i <= total; i++) {
                    addPage(i);
                }
                return pages;
            }

            addPage(1);

            if (current <= 4) {
                for (let i = 2; i <= 5; i++) {
                    addPage(i);
                }
                addEllipsis('right');
            } else if (current >= total - 3) {
                addEllipsis('left');
                for (let i = total - 4; i < total; i++) {
                    addPage(i);
                }
            } else {
                addEllipsis('left');
                for (let i = current - 1; i <= current + 1; i++) {
                    addPage(i);
                }
                addEllipsis('right');
            }

            addPage(total);
            return pages;
        });

        const filtered = computed(() => {
            const term = search.value.trim().toLowerCase();
            let list = transactions.value;
            if (term) {
                list = list.filter(tx => {
                    return (tx.name || '').toLowerCase().includes(term)
                        || (tx.date || '').toLowerCase().includes(term)
                        || String(tx.amount || '').includes(term);
                });
            }
            if (sortKey.value) {
                list = [...list].sort((a, b) => {
                    const dir = sortDir.value === 'asc' ? 1 : -1;
                    const va = a[sortKey.value] ?? '';
                    const vb = b[sortKey.value] ?? '';
                    if (typeof va === 'number' && typeof vb === 'number') {
                        return (va - vb) * dir;
                    }
                    return String(va).localeCompare(String(vb)) * dir;
                });
            }
            return list;
        });

        const totalPages = computed(() => Math.max(1, Math.ceil(totalTransactions.value / pageSize.value)));
        const pageStart = computed(() => (currentPage.value - 1) * pageSize.value);
        const paged = computed(() => filtered.value); 

        const accountMap = computed(() => {
            const map = {};
            (accounts.value || []).forEach(acc => {
                if (acc && acc.account_id) {
                    map[acc.account_id] = {
                        name: acc.name || '',
                        subtype: acc.subtype || ''
                    };
                }
            });
            return map;
        });

        const accountName = (accountId) => (accountMap.value[accountId] && accountMap.value[accountId].name) || '';
        const accountSubtype = (accountId) => (accountMap.value[accountId] && accountMap.value[accountId].subtype) || '-';

        const formatApiDate = (date) => {
            if (!date) return null;
            const year = date.getFullYear();
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const day = String(date.getDate()).padStart(2, '0');
            return year + '-' + month + '-' + day;
        };

        const formatDisplayDate = (date) => {
            if (!date) return '';
            return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
        };

        const setRangeLabel = (dates) => {
            if (!dates || dates.length < 2) {
                rangeLabel.value = 'Select range';
                return;
            }
            rangeLabel.value = formatDisplayDate(dates[0]) + ' - ' + formatDisplayDate(dates[1]);
        };

        const openDatepicker = async () => {
            await ensureRangePicker();
            if (rangePicker) {
                rangePicker.open();
            }
        };

        const applyRange = (startDate, endDate, rangeKey) => {
            currentRange.value = rangeKey;
            currentStart.value = startDate;
            currentEnd.value = endDate;
            setRangeLabel([startDate, endDate]);
            if (rangePicker) {
                rangePicker.setDate([startDate, endDate], true);
            }
            currentPage.value = 1;
            getTransactions();
        };

        const setPresetRange = (rangeKey) => {
            const endDate = new Date();
            let startDate = new Date(endDate);
            if (rangeKey === 'weekly') {
                const day = startDate.getDay();
                const diffToMonday = (day === 0 ? 6 : day - 1);
                startDate.setDate(startDate.getDate() - diffToMonday);
            } else if (rangeKey === 'monthly') {
                startDate = new Date(endDate.getFullYear(), endDate.getMonth(), 1);
            } else if (rangeKey === 'yearly') {
                startDate = new Date(endDate.getFullYear(), 0, 1);
            }
            applyRange(startDate, endDate, rangeKey);
        };

        const initRangePicker = () => {
            if (typeof flatpickr === 'undefined' || !dateRangeInput.value) {
                return;
            }

            const endDate = new Date();
            const startDate = new Date(endDate);
            startDate.setFullYear(startDate.getFullYear() - 2);

            rangePicker = flatpickr(dateRangeInput.value, {
                mode: 'range',
                defaultDate: [startDate, endDate],
                dateFormat: 'M j, Y',
                onReady: function (selectedDates) { setRangeLabel(selectedDates); },
                onChange: function (selectedDates) {
                    setRangeLabel(selectedDates);
                    if (selectedDates.length === 2) {
                        currentRange.value = 'custom';
                        currentStart.value = selectedDates[0];
                        currentEnd.value = selectedDates[1];
                        search.value = '';
                        currentPage.value = 1;
                        getTransactions();
                    }
                }
            });

            currentRange.value = 'weekly';
            currentStart.value = startDate;
            currentEnd.value = endDate;
            setRangeLabel([startDate, endDate]);
        };

        const ensureRangePicker = async () => {
            if (rangePicker) return;
            await nextTick();
            if (!dateRangeInput.value) return;
            initRangePicker();
        };

        const updateInstitutionForSelected = () => {
            if (!selectedConnection.value) return;
            const found = connections.value.find(c => c.id === selectedConnection.value);
            if (found) {
                institution.value = found.institution_name || null;
            }
        };

        const toggleSort = (key) => {
            if (sortKey.value === key) {
                sortDir.value = sortDir.value === 'asc' ? 'desc' : 'asc';
            } else {
                sortKey.value = key;
                sortDir.value = 'asc';
            }
        };

        const goToPage = (page) => {
            if (page < 1) page = 1;
            if (page > totalPages.value) page = totalPages.value;
            currentPage.value = page;
            getTransactions(); 
        };

        const ensurePageInRange = () => {
            if (currentPage.value > totalPages.value) {
                currentPage.value = totalPages.value;
            }
        };

        const fetchStatus = async () => {
            try {
                const res = await fetch(
                    '{$app_url}api/plaid/connection/status' +
                    (cid ? '?client_id=' + encodeURIComponent(cid) : '')
                );
                if (!res.ok) {
                    connected.value = false;
                    toastr.error('Connection error');
                    return;
                }
                const data = await res.json();
                connected.value = !!data.connected;
                institution.value = data.institution_name || null;
                connections.value = Array.isArray(data.connections) ? data.connections : [];
                if (!selectedConnection.value && data.active_connection_id) {
                    selectedConnection.value = data.active_connection_id;
                } else if (!selectedConnection.value && connections.value.length) {
                    selectedConnection.value = connections.value[0].id;
                }                
                updateInstitutionForSelected();
            } catch (err) {
                console.error('No se pudo obtener el estado de Plaid', err);
                connected.value = false;
            }
        };

        const handleTransactionsResponse = (data) => {
            if (!data || typeof data !== 'object') {
                toastr.error('No se pudo leer la respuesta de transacciones. Intenta nuevamente.');
                return false;
            }
            if (data.error_code === 'ITEM_LOGIN_REQUIRED') {
                toastr.error('Tu banco requiere que vuelvas a conectarte. Presiona "Connect bank account" para actualizar el acceso.');
                connected.value = false;
                return false;
            }
            if (data.error || data.error_message || data.error_type) {
                const msg = data.error_message || data.display_message || data.error || 'Error al obtener transacciones.';
                toastr.error(msg);
                return false;
            }
            if (!Array.isArray(data.transactions)) {
                toastr.error('No se recibieron transacciones en la respuesta.');
                return false;
            }
            return true;
        };

        const getTransactions = async () => {
            loading.value = true;
            updateInstitutionForSelected();

            const offset = (currentPage.value - 1) * pageSize.value;

            console.log('page', currentPage.value, 'count', pageSize.value, 'offset', offset);

            try {
                const payload = {
                    client_id: cid,
                    connection_id: selectedConnection.value,
                    count: pageSize.value,
                    offset
                };

                if (currentStart.value && currentEnd.value) {
                    payload.start_date = formatApiDate(currentStart.value);
                    payload.end_date = formatApiDate(currentEnd.value);
                }

                const response = await fetch('{$app_url}api/plaid/transactions', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(payload)
                });

                const data = await response.json();
                if (!handleTransactionsResponse(data)) return;

                transactions.value = data.transactions || [];
                accounts.value = data.accounts || [];
                totalTransactions.value = data.total_transactions ?? 0;
                ensurePageInRange();

            } catch (err) {
                console.error(err);
                toastr.error('No se pudieron obtener transacciones. Intenta nuevamente.');
            } finally {
                loading.value = false;
            }
        };


        const connect = async () => {
            loading.value = true;
            try {
                const res = await fetch(
                    '{$app_url}api/plaid/create_link_token' +
                    (cid ? '?client_id=' + encodeURIComponent(cid) : ''),
                    {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' }
                    }
                );
                if (!res.ok) {
                    const err = await res.json().catch(() => ({}));
                    toastr.error('Error obteniendo link token: ' + (err.error || res.status));
                    return;
                }
                const body = await res.json();
                const linkToken = body.link_token;
                if (!linkToken) {
                    const msg = res?.message || 'Invalid link token.';
                    toastr.error(msg);
                    return;
                }

                const handler = Plaid.create({
                    token: linkToken,
                    onSuccess: async function(public_token) {
                        try {
                            const exchangeRes = await fetch('{$app_url}api/plaid/exchange_public_token', {
                                method: 'POST',
                                headers: { 'Content-Type': 'application/json' },
                                body: JSON.stringify({ public_token, client_id: cid })
                            });
                            const exchangeBody = await exchangeRes.json();
                            if (!exchangeRes.ok) {
                                toastr.error(exchangeBody.error || 'Connection error');
                                return;
                            }
                            toastr.success('Connected account');
                            await fetchStatus();
                            if (connected.value) {
                                await ensureRangePicker();
                                getTransactions();
                            }
                        } catch (err) {
                            console.error(err);
                            toastr.error('Connection error');
                        }
                    },
                    onExit: function(err) {
                        if (err) {
                            console.error('Plaid Link exit with error:', err);
                        }
                    },
                    onEvent: function(eventName, metadata) {
                        console.log('Plaid event', eventName, metadata);
                    }
                });
                handler.open();
            } catch (err) {
                console.error(err);
                toastr.error('Plaid Link error: ' + err.message);
            } finally {
                loading.value = false;
            }
        };

        const handleConnectionChange = () => {
            search.value = '';
            currentPage.value = 1;
            getTransactions();
        };

        watch(pageSize, () => {
            currentPage.value = 1;
            getTransactions();
        });

        onMounted(async () => {
            const params = new URLSearchParams(window.location.search || '');
            const autoPlaid = params.get('plaid') === '1';

            await fetchStatus();
            if (connected.value) {
                await ensureRangePicker();
                getTransactions();
            } else if (autoPlaid) {
                connect();
            }
        });

        return {
            loading,
            connected,
            institution,
            connections,
            selectedConnection,
            transactions,
            search,
            filtered,
            paged,
            pageSize,
            currentPage,
            totalPages,
            pageStart,
            toggleSort,
            goToPage,
            pageNumbers,
            totalTransactions,
            sortKey,
            sortDir,
            connect,
            handleConnectionChange,
            getTransactions,
            accountName,
            accountSubtype,
            dateRangeInput,
            dateDisplayBtn,
            rangeLabel,
            currentRange,
            setPresetRange,
            openDatepicker
        };
    }
}).mount('#plaid-app');
</script>
{/block}
