<style>
[v-cloak] { display: none; }
</style>

<div id="auto-payment-app" class="client-view" v-cloak>

    <header class="client-header">
        <h1>Stripe Subscriptions</h1>
        <p>Create and manage recurring subscriptions with Stripe Billing.</p>
    </header>

    <div class="client-body">

        <div class="block">
            <div class="block-header">
                <h2 class="block-title">Create Subscription</h2>
            </div>
            <div class="block-body">
                <form @submit.prevent="createSubscription">
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Customer</label>
                            <input type="text" class="form-control" :value="contactName" readonly>
                        </div>
                        <div class="form-group">
                            <label>Email</label>
                            <input type="text" class="form-control" :value="contactEmail" readonly>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label>CRM Invoice <span class="text-danger">*</span></label>
                            <select v-model="selectedInvoiceId" class="form-control" required>
                                <option value="">Select an invoice</option>
                                <option v-for="inv in invoices" :key="inv.id" :value="inv.id">
                                    {{ formatInvoiceLabel(inv) }}
                                </option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Charge Type</label>
                            <select v-model="chargeType" class="form-control">
                                <option value="recurring">Recurring</option>
                                <option value="one_time">One-time future charge</option>
                            </select>
                        </div>
                    </div>

                    <div class="form-row" v-if="chargeType === 'recurring'">
                        <div class="form-group">
                            <label>Frequency</label>
                            <select v-model="frequency" class="form-control">
                                <option value="monthly">Monthly</option>
                                <option value="weekly">Weekly</option>
                                <option value="bi-weekly">Bi-weekly</option>
                                <option value="quarterly">Quarterly</option>
                                <option value="yearly">Yearly</option>
                            </select>
                        </div>
                    </div>

                    <div class="form-row" v-if="chargeType === 'one_time'">
                        <div class="form-group">
                            <label>Charge Date <span class="text-danger">*</span></label>
                            <input type="date" v-model="chargeDate" class="form-control" required>
                        </div>
                        <div class="form-group">
                            <label>Charge Time <span class="text-danger">*</span></label>
                            <input type="time" v-model="chargeTime" class="form-control" required>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label>Amount</label>
                            <input type="text" class="form-control" :value="selectedInvoiceAmount" readonly>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label>
                                <input type="checkbox" v-model="useCustomAmount">
                                Use custom amount
                            </label>
                        </div>
                    </div>

                    <div class="form-row" v-if="useCustomAmount">
                        <div class="form-group">
                            <label>Custom Amount <span class="text-danger">*</span></label>
                            <input type="number" v-model="customAmount" class="form-control" step="0.01" min="0.01" placeholder="0.00">
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label>Payment Method <span class="text-danger">*</span></label>
                            <select v-model="selectedPaymentMethodId" class="form-control" required>
                                <option value="">Select a payment method</option>
                                <option v-for="pm in paymentMethods" :key="pm.id" :value="pm.payment_method" :disabled="!isPaymentMethodUsable(pm)">
                                    {{ formatPaymentMethodLabel(pm) }}
                                </option>
                            </select>
                            <small v-if="paymentMethods.length === 0" class="text-muted">
                                No saved payment methods. Add one in Payment Methods first.
                            </small>
                            <small v-else-if="hasUnverifiedMethods" class="text-muted">
                                Bank accounts must be verified before use.
                            </small>
                        </div>
                    </div>

                    <div class="form-actions">
                        <button type="submit" class="ui-button primary" :disabled="submitting">
                            Create Subscription
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <div class="block w-table">
            <div class="block-header">
                <h2 class="block-title">Subscriptions</h2>
            </div>
            
            <div class="block-body">
                <div v-if="loading && subscriptions.length === 0" class="loading-state">
                    <div>Loading...</div>
                </div>
                
                <div v-else-if="subscriptions.length === 0" class="empty-state">
                    <i class="fal fa-calendar-times"></i>
                    <p>No subscriptions</p>
                </div>

                <div v-else>
                    <table class="ui-table">
                        <thead>
                            <tr>
                                <th>STATUS</th>
                                <th>CRM INVOICE</th>
                                <th>AMOUNT</th>
                                <th>CRM STATUS</th>
                                <th>INTERVAL</th>
                                <th>RENEWS</th>
                                <th>ACTIONS</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr v-for="sub in subscriptions" :key="sub.id">
                                <td data-label="Status">
                                    <span class="status-badge" :class="sub.status">
                                        <span class="status-dot" :class="sub.status"></span>
                                        {{ sub.status }}
                                    </span>
                                </td>
                                <td data-label="CRM Invoice"><strong>{{ formatInvoiceReference(sub) }}</strong></td>
                                <td data-label="Amount"><strong>{{ formatAmountCents(sub.amount, sub.currency) }}</strong></td>
                                <td data-label="CRM Status">{{ sub.crm_invoice_status || '---' }}</td>
                                <td data-label="Interval"><span class="frequency-badge">{{ sub.interval || '---' }}</span></td>
                                <td data-label="Renews">
                                    <span v-if="sub.interval === 'one_time'">
                                        Scheduled: {{ formatTimestamp(sub.current_period_end) }}
                                    </span>
                                    <span v-else>
                                        {{ formatTimestamp(sub.current_period_end) }}
                                    </span>
                                </td>
                                <td data-actions>
                                    <button v-if="canCancel(sub)" class="ui-button secondary" @click="cancelSubscription(sub)" :disabled="submitting">
                                        Cancel
                                    </button>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                    <div class="table-footer">
                        Showing {{ subscriptions.length }} subscription{{ subscriptions.length !== 1 ? 's' : '' }}
                    </div>
                </div>
            </div>

        </div>

    </div>

</div>

<script>
if (typeof Vue !== 'undefined') {
    const autoPaymentApp = Vue.createApp({
        data() {
            return {
                contactId: null,
                contactName: 'Loading...',
                contactEmail: '',
                appUrl: '',
                invoices: [],
                subscriptions: [],
                paymentMethods: [],
                stripeCustomerId: '',
                publishableKey: '',
                stripe: null,
                selectedPaymentMethodId: '',
                stripeDefaultPaymentMethodId: '',
                selectedInvoiceId: '',
                frequency: 'monthly',
                chargeType: 'recurring',
                chargeDate: new Date().toISOString().split('T')[0],
                chargeTime: '09:00',
                useCustomAmount: false,
                customAmount: '',
                loading: false,
                submitting: false
            };
        },
        computed: {
            selectedInvoiceAmount() {
                if (this.useCustomAmount && this.customAmount) {
                    return '$' + this.formatNumber(this.customAmount);
                }
                const inv = this.invoices.find(i => String(i.id) === String(this.selectedInvoiceId));
                if (!inv || !inv.total) return '---';
                return '$' + this.formatNumber(inv.total);
            }
        },
        mounted() {
            this.appUrl = (document.getElementById('_app_url') || {}).value || '';
            this.contactId = this.extractContactId();
            this.initialize();
        },
        methods: {
            extractContactId() {
                const url = new URL(window.location.href);
                const ngParam = url.searchParams.get('ng');
                return ngParam ? ngParam.split('/')[2] : null;
            },
            async initialize() {
                this.loading = true;
                try {
                    await this.loadContact();
                    await this.loadConfig();
                    await this.loadPaymentMethods();
                    await this.ensureStripeCustomer();
                    await this.loadInvoices();
                    await this.loadStripeDefaultPaymentMethod();
                    await this.loadSubscriptions();
                    this.ensureStripeInstance();
                } catch (error) {
                    console.error('Initialization error:', error);
                    if (typeof toastr !== 'undefined') toastr.error('Unable to initialize Stripe subscriptions');
                } finally {
                    this.loading = false;
                }
            },
            async loadContact() {
                const response = await fetch(this.appUrl + '/api/contacts/summary/' + this.contactId, {
                    method: 'GET'
                });
                const data = await response.json();
                if (data.customer) {
                    this.contactName = data.customer.account || 'Unknown Contact';
                    this.contactEmail = data.customer.email || '';
                }
            },
            async loadConfig() {
                const response = await fetch(this.appUrl + '/api/billing/config', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({})
                });
                const data = await response.json();
                this.publishableKey = data.publishable_key || '';
            },
            ensureStripeInstance() {
                if (this.publishableKey && !this.stripe && typeof Stripe !== 'undefined') {
                    this.stripe = Stripe(this.publishableKey);
                }
            },
            async ensureStripeCustomer() {
                if (this.isStripeCustomerId(this.stripeCustomerId)) {
                    return;
                }
                const response = await fetch(this.appUrl + '/api/billing/customers', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        crm_customer_id: this.contactId,
                        email: this.contactEmail,
                        name: this.contactName
                    })
                });
                const data = await response.json();
                this.stripeCustomerId = data.stripe_customer_id || '';
            },
            async loadInvoices() {
                const response = await fetch(this.appUrl + '/api/contacts/invoices', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: new URLSearchParams({ cid: this.contactId })
                });
                const data = await response.json();
                if (data.status === 'success') {
                    this.invoices = data.data || [];
                } else {
                    this.invoices = [];
                }
            },
            async loadPaymentMethods() {
                const response = await fetch(this.appUrl + '/api/contacts/payment-methods/table', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: new URLSearchParams({ cid: this.contactId })
                });
                const data = await response.json();
                if (data.status === 'success') {
                    this.paymentMethods = (data.data || []).filter(pm => pm.payment_method);
                    this.syncStripeCustomerFromPaymentMethods();
                } else {
                    this.paymentMethods = [];
                }
            },
            syncStripeCustomerFromPaymentMethods() {
                if (this.isStripeCustomerId(this.stripeCustomerId)) {
                    return;
                }
                const match = this.paymentMethods.find(pm => this.isStripeCustomerId(pm.customer_id || ''));
                if (match) {
                    this.stripeCustomerId = match.customer_id;
                }
            },
            isPaymentMethodUsable(pm) {
                if (!pm) return false;
                if (String(pm.tipo_cuenta) === '3') return true;
                return String(pm.verificado) === '1';
            },
            isStripeCustomerId(value) {
                return typeof value === 'string' && value.startsWith('cus_');
            },
            hasUnverifiedMethods() {
                return this.paymentMethods.some(pm => String(pm.tipo_cuenta) !== '3' && String(pm.verificado) !== '1');
            },
            async loadStripeDefaultPaymentMethod() {
                if (!this.stripeCustomerId) return;
                const response = await fetch(this.appUrl + '/api/billing/payment-methods/default/get', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        stripe_customer_id: this.stripeCustomerId
                    })
                });
                const data = await response.json();
                this.stripeDefaultPaymentMethodId = data.payment_method_id || '';
                if (this.stripeDefaultPaymentMethodId && !this.selectedPaymentMethodId) {
                    this.selectedPaymentMethodId = this.stripeDefaultPaymentMethodId;
                }
            },
            async loadSubscriptions() {
                if (!this.stripeCustomerId) return;
                const response = await fetch(this.appUrl + '/api/billing/subscriptions/list', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        stripe_customer_id: this.stripeCustomerId
                    })
                });
                const data = await response.json();
                this.subscriptions = data.data || [];
            },
            canCancel(sub) {
                if (!sub) return false;
                const status = String(sub.status || '').toLowerCase();
                if (status === 'cancelled' || status === 'canceled' || status === 'incomplete_expired' || status === 'paid') {
                    return false;
                }
                if (sub.interval === 'one_time' && status !== 'scheduled') {
                    return false;
                }
                return true;
            },
            async cancelSubscription(sub) {
                if (!sub || !sub.id) {
                    return;
                }

                const isOneTime = sub.interval === 'one_time';
                const text = isOneTime
                    ? 'Cancel this scheduled invoice?'
                    : 'Cancel this subscription at period end?';
                let useImmediate = false;

                if (window.Swal && Swal.fire) {
                    if (isOneTime) {
                        const result = await Swal.fire({
                            title: 'Confirm',
                            text,
                            icon: 'warning',
                            showCancelButton: true,
                            confirmButtonText: 'Yes, cancel',
                            cancelButtonText: 'Keep'
                        });
                        if (!result.isConfirmed) {
                            return;
                        }
                    } else {
                        const result = await Swal.fire({
                            title: 'Cancel subscription',
                            text: 'Do you want to cancel this subscription?',
                            icon: 'warning',
                            showCancelButton: true,
                            confirmButtonText: 'Yes, cancel',
                            cancelButtonText: 'Keep'
                        });
                        if (!result.isConfirmed) {
                            return;
                        }

                        const immediateResult = await Swal.fire({
                            title: 'Cancel timing',
                            text: 'Cancel immediately? (If not, it will cancel at period end.)',
                            icon: 'question',
                            showCancelButton: true,
                            confirmButtonText: 'Cancel immediately',
                            cancelButtonText: 'Cancel at period end'
                        });
                        if (immediateResult.isConfirmed) {
                            useImmediate = true;
                        }
                    }
                } else if (!confirm(text)) {
                    if (isOneTime) {
                        return;
                    }
                    const immediate = confirm('Cancel immediately? Click OK to cancel now, Cancel for period end.');
                    if (immediate) {
                        useImmediate = true;
                    } else if (!confirm('Cancel at period end?')) {
                        return;
                    }
                }

                this.submitting = true;
                try {
                    const endpoint = (!isOneTime && useImmediate)
                        ? '/api/billing/subscriptions/cancel-immediate'
                        : '/api/billing/subscriptions/cancel';
                    const response = await fetch(this.appUrl + endpoint, {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            stripe_id: sub.id,
                            kind: isOneTime ? 'invoice' : 'subscription'
                        })
                    });
                    const data = await response.json();
                    if (!response.ok || data.error) {
                        if (typeof toastr !== 'undefined') {
                            toastr.error(data.error || 'Unable to cancel');
                        }
                        return;
                    }
                    if (typeof toastr !== 'undefined') {
                        toastr.success('Cancelled');
                    }
                    await this.loadSubscriptions();
                } catch (error) {
                    console.error('Cancel error:', error);
                    if (typeof toastr !== 'undefined') {
                        toastr.error('Unable to cancel');
                    }
                } finally {
                    this.submitting = false;
                }
            },
            async createSubscription() {
                if (!this.selectedInvoiceId) {
                    if (typeof toastr !== 'undefined') toastr.error('Please select an invoice');
                    return;
                }
                if (!this.selectedPaymentMethodId) {
                    if (typeof toastr !== 'undefined') toastr.error('Please select a payment method');
                    return;
                }
                if (this.chargeType === 'one_time' && !this.chargeDate) {
                    if (typeof toastr !== 'undefined') toastr.error('Please select a charge date');
                    return;
                }
                if (this.chargeType === 'one_time' && !this.chargeTime) {
                    if (typeof toastr !== 'undefined') toastr.error('Please select a charge time');
                    return;
                }
                if (this.useCustomAmount) {
                    const amountValue = parseFloat(this.customAmount);
                    if (Number.isNaN(amountValue) || amountValue <= 0) {
                        if (typeof toastr !== 'undefined') toastr.error('Please enter a valid custom amount');
                        return;
                    }
                }

                this.submitting = true;
                try {
                    await fetch(this.appUrl + '/api/billing/payment-methods/default', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            stripe_customer_id: this.stripeCustomerId,
                            payment_method_id: this.selectedPaymentMethodId
                        })
                    });

                    const response = await fetch(this.appUrl + '/api/billing/subscriptions', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            stripe_customer_id: this.stripeCustomerId,
                            crm_invoice_id: this.selectedInvoiceId,
                            frequency: this.frequency,
                            charge_type: this.chargeType,
                            charge_date: this.chargeDate,
                            charge_time: this.chargeTime,
                            custom_amount: this.useCustomAmount ? this.customAmount : ''
                        })
                    });
                    const data = await response.json();
                    if (!response.ok || data.error) {
                        if (typeof toastr !== 'undefined') {
                            toastr.error(data.error || 'Subscription failed');
                        }
                        return;
                    }

                    if (data.status === 'scheduled') {
                        if (typeof toastr !== 'undefined') toastr.success('Charge scheduled');
                        await this.loadSubscriptions();
                        return;
                    }

                    const selected = this.findSelectedPaymentMethod();
                    const isCard = selected && String(selected.tipo_cuenta) === '3';

                    if (data.client_secret && isCard) {
                        this.ensureStripeInstance();
                        if (this.stripe) {
                            const confirmResult = await this.stripe.confirmCardPayment(
                                data.client_secret,
                                { payment_method: this.selectedPaymentMethodId }
                            );
                            if (confirmResult.error) {
                                if (typeof toastr !== 'undefined') toastr.error(confirmResult.error.message || 'Payment confirmation failed');
                                return;
                            }
                        }
                    }

                    if (typeof toastr !== 'undefined') toastr.success('Subscription created');
                    await this.loadSubscriptions();
                } catch (error) {
                    console.error('Subscription error:', error);
                    if (typeof toastr !== 'undefined') toastr.error('Subscription failed');
                } finally {
                    this.submitting = false;
                }
            },
            formatInvoiceLabel(inv) {
                const display = inv.invoice_display || inv.invoicenum || inv.id;
                const number = display ? ('#' + display) : ('#' + inv.id);
                const total = inv.total ? this.formatNumber(inv.total) : '0.00';
                const status = inv.status || 'unknown';
                return number + ' - $' + total + ' - ' + status;
            },
            formatPaymentMethodLabel(pm) {
                const type = pm.nombre_tipo_cuenta || pm.tipo_pasarela_pago || 'Payment method';
                const last4 = pm.last_4_digits || (pm.numero_tarjeta ? String(pm.numero_tarjeta).slice(-4) : '');
                const bank = pm.banco_metodo_pago || '';
                const parts = [type];
                if (bank) parts.push(bank);
                if (last4) parts.push('**** ' + last4);
                return parts.join(' - ');
            },
            findSelectedPaymentMethod() {
                return this.paymentMethods.find(pm => pm.payment_method === this.selectedPaymentMethodId) || null;
            },
            formatInvoiceReference(sub) {
                if (sub.crm_invoice_id) {
                    return 'CRM #' + sub.crm_invoice_id;
                }
                return sub.id || '---';
            },
            formatNumber(value) {
                const amount = parseFloat(value);
                if (Number.isNaN(amount)) return '0.00';
                return amount.toFixed(2);
            },
            formatAmountCents(amount, currency) {
                if (!amount) return '---';
                const formatted = (amount / 100).toFixed(2);
                const code = (currency || 'usd').toUpperCase();
                return code + ' ' + formatted;
            },
            formatTimestamp(unixSeconds) {
                if (!unixSeconds) return '---';
                const date = new Date(unixSeconds * 1000);
                return date.toLocaleString('en-US', {
                    year: 'numeric',
                    month: 'short',
                    day: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
            }
        }
    });

    autoPaymentApp.mount('#auto-payment-app');
}
</script>
