{block name="content"}
<div id="plaid-app" v-cloak>

    <div class="row">
        <div class="col-lg-12 mt-3">
            
            <h2>Finance</h2>

            <div class="finance-connected">
                <div class="finance-connected-bar">
                    <div class="finance-connected-info">
                        <template v-if="connected">
                            <div class="connected-chip">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="20 6 9 17 4 12"></polyline>
                                </svg>
                                Connected
                            </div>
                            <div>{{ institution || 'Linked bank account' }}</div>
                        </template>
                        <template v-else>
                            <div class="connected-chip-warning">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <line x1="5" y1="5" x2="19" y2="19"></line>
                                    <line x1="19" y1="5" x2="5" y2="19"></line>
                                </svg>
                                Not connected
                            </div>
                        </template>
                    </div>
                    <div class="finance-actions">
                        <input type="search" v-model="search" class="form-control input-search-table finance-search" placeholder="Search">
                        <select class="form-control finance-search" v-model.number="selectedConnection" @change="getTransactions" style="max-width: 240px;">
                            <option disabled :value="null">Select bank account</option>
                            <option v-for="conn in connections" :key="conn.id" :value="conn.id">
                                {{ conn.institution_name || ('Account ' + conn.id) }}
                            </option>
                        </select>
                        <button type="button" class="button" @click="getTransactions" :disabled="loading">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <polyline points="23 4 23 10 17 10"></polyline>
                                <polyline points="1 20 1 14 7 14"></polyline>
                                <path d="M3.51 9a9 9 0 0 1 14.85-3.36L23 10"></path>
                                <path d="M20.49 15a9 9 0 0 1-14.85 3.36L1 14"></path>
                            </svg>
                            {{ loading ? 'Refreshing...' : 'Refresh' }}
                        </button>
                    </div>
                </div>

                <div class="table-container tc-table mt-3">
                    <div v-if="loading" class="skeleton-table">
                        <div class="skeleton-row" v-for="n in 6" :key="n">
                            <div class="skeleton-cell" style="width: 120px;"></div>
                            <div class="skeleton-cell" style="width: 60%;"></div>
                            <div class="skeleton-cell" style="width: 40%;"></div>
                            <div class="skeleton-cell" style="width: 50%;"></div>
                            <div class="skeleton-cell" style="width: 80px;"></div>
                        </div>
                    </div>
                    <table v-else id="transactions">
                        <thead>
                            <tr>
                                <th @click="toggleSort('date')" role="button">
                                    Date
                                    <svg class="sort-icon" :class="sortKey === 'date' ? (sortDir === 'asc' ? 'is-asc' : 'is-desc') : 'is-neutral'" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
                                        <path d="M12 16l-6-6h12z"></path>
                                    </svg>
                                </th>
                                <th @click="toggleSort('name')" role="button">
                                    Description
                                    <svg class="sort-icon" :class="sortKey === 'name' ? (sortDir === 'asc' ? 'is-asc' : 'is-desc') : 'is-neutral'" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
                                        <path d="M12 16l-6-6h12z"></path>
                                    </svg>
                                </th>
                                <th class="text-start" @click="toggleSort('amount')" role="button">
                                    Amount
                                    <svg class="sort-icon" :class="sortKey === 'amount' ? (sortDir === 'asc' ? 'is-asc' : 'is-desc') : 'is-neutral'" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
                                        <path d="M12 16l-6-6h12z"></path>
                                    </svg>
                                </th>
                                <th>Account Name</th>
                                <th>Account Type</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr v-if="!paged.length && !loading" height="180">
                                <td colspan="5" class="text-center">No transactions found.</td>
                            </tr>
                            <tr v-for="tx in paged" :key="tx.transaction_id || (tx.name + tx.date)">
                                <td data-label="Date">{{ tx.date }}</td>
                                <td data-label="Description">{{ tx.name }}</td>
                                <td data-label="Amount" class="text-start" :class="tx.amount < 0 ? 'text-positive' : 'text-negative'">
                                    {{ (tx.amount || 0) * -1 }} {{ tx.iso_currency_code }}
                                </td>
                                <td data-label="Account Name">{{ accountName(tx.account_id) }}</td>
                                <td data-label="Account Type">{{ accountSubtype(tx.account_id) }}</td>
                            </tr>
                        </tbody>
                    </table>
                </div>

                <div class="table-footer mt-2" v-if="filtered.length">
                    <div>Showing {{ pageStart + 1 }} to {{ Math.min(pageStart + pageSize, filtered.length) }} of {{ filtered.length }} entries</div>
                    <div class="d-flex align-items-center gap-2">
                        <select v-model.number="pageSize" class="form-control select-per-page">
                            <option :value="10">10 per page</option>
                            <option :value="25">25 per page</option>
                            <option :value="50">50 per page</option>
                        </select>
                        <div class="table-paging">
                            <button class="button" @click="goToPage(currentPage - 1)" :disabled="currentPage === 1"><</button>
                            <template v-for="item in pageNumbers" :key="item.key">
                                <button
                                    v-if="item.type === 'page'"
                                    class="button"
                                    :class="item.value === currentPage ? 'button-active-page' : ''"
                                    @click="goToPage(item.value)"
                                >
                                    {{ item.value }}
                                </button>
                                <button
                                    v-else
                                    class="button button-ellipsis"
                                    type="button"
                                    disabled
                                    aria-hidden="true"
                                >
                                    ...
                                </button>
                            </template>
                            <button class="button" @click="goToPage(currentPage + 1)" :disabled="currentPage >= totalPages">></button>
                        </div>
                    </div>
                </div>
            </div>
            
        </div>
    </div>

</div>
{/block}

{block name="script"}
<script>
// Vue helpers scoped per-load to avoid re-declare errors when tab reloads
(() => {
    const createApp = Vue.createApp;
    const ref = Vue.ref;
    const computed = Vue.computed;
    const onMounted = Vue.onMounted;
    const appUrl = '{$app_url}';
    const cid = '{$cid|default:""}';

{literal}
createApp({
    setup() {
        const loading = ref(false);
        const connected = ref(false);
        const institution = ref(null);
        const connections = ref([]);
        const selectedConnection = ref(null);
        const transactions = ref([]);
        const accounts = ref([]);
        const search = ref('');
        const sortKey = ref(null);
        const sortDir = ref('asc');
        const pageSize = ref(10);
        const currentPage = ref(1);

        const pageNumbers = computed(() => {
            const total = totalPages.value;
            const current = currentPage.value;
            const pages = [];

            const addPage = (page) => pages.push({ type: 'page', value: page, key: 'p-' + page });
            const addEllipsis = (id) => pages.push({ type: 'ellipsis', value: id, key: 'e-' + id + '-' + total + '-' + current });

            if (total <= 7) {
                for (let i = 1; i <= total; i++) {
                    addPage(i);
                }
                return pages;
            }

            addPage(1);

            if (current <= 4) {
                for (let i = 2; i <= 5; i++) {
                    addPage(i);
                }
                addEllipsis('right');
            } else if (current >= total - 3) {
                addEllipsis('left');
                for (let i = total - 4; i < total; i++) {
                    addPage(i);
                }
            } else {
                addEllipsis('left');
                for (let i = current - 1; i <= current + 1; i++) {
                    addPage(i);
                }
                addEllipsis('right');
            }

            addPage(total);
            return pages;
        });

        const filtered = computed(() => {
            const term = search.value.trim().toLowerCase();
            let list = transactions.value;
            if (term) {
                list = list.filter(tx => {
                    return (tx.name || '').toLowerCase().includes(term)
                        || (tx.date || '').toLowerCase().includes(term)
                        || String(tx.amount || '').includes(term);
                });
            }
            if (sortKey.value) {
                list = [...list].sort((a, b) => {
                    const dir = sortDir.value === 'asc' ? 1 : -1;
                    const va = a[sortKey.value] ?? '';
                    const vb = b[sortKey.value] ?? '';
                    if (typeof va === 'number' && typeof vb === 'number') {
                        return (va - vb) * dir;
                    }
                    return String(va).localeCompare(String(vb)) * dir;
                });
            }
            return list;
        });

        const totalPages = computed(() => Math.max(1, Math.ceil(filtered.value.length / pageSize.value)));
        const pageStart = computed(() => (currentPage.value - 1) * pageSize.value);
        const paged = computed(() => filtered.value.slice(pageStart.value, pageStart.value + pageSize.value));

        const accountMap = computed(() => {
            const map = {};
            (accounts.value || []).forEach(acc => {
                if (acc && acc.account_id) {
                    map[acc.account_id] = {
                        name: acc.name || '',
                        subtype: acc.subtype || ''
                    };
                }
            });
            return map;
        });

        const accountName = (accountId) => (accountMap.value[accountId] && accountMap.value[accountId].name) || '';
        const accountSubtype = (accountId) => (accountMap.value[accountId] && accountMap.value[accountId].subtype) || '-';

        const updateInstitutionForSelected = () => {
            if (!selectedConnection.value) return;
            const found = connections.value.find(c => c.id === selectedConnection.value);
            if (found) {
                institution.value = found.institution_name || null;
            }
        };

        const toggleSort = (key) => {
            if (sortKey.value === key) {
                sortDir.value = sortDir.value === 'asc' ? 'desc' : 'asc';
            } else {
                sortKey.value = key;
                sortDir.value = 'asc';
            }
        };

        const goToPage = (page) => {
            if (page < 1) page = 1;
            if (page > totalPages.value) page = totalPages.value;
            currentPage.value = page;
        };

        const ensurePageInRange = () => {
            if (currentPage.value > totalPages.value) {
                currentPage.value = totalPages.value;
            }
        };

        const fetchStatus = async () => {
            try {
                const res = await fetch(`${appUrl}api/plaid/connection/status${cid ? '?client_id=' + encodeURIComponent(cid) : ''}`);
                if (!res.ok) {
                    connected.value = false;
                    toastr.error('Connection error');
                    return;
                }
                const data = await res.json();
                connected.value = !!data.connected;
                institution.value = data.institution_name || null;
                connections.value = Array.isArray(data.connections) ? data.connections : [];
                if (!selectedConnection.value && data.active_connection_id) {
                    selectedConnection.value = data.active_connection_id;
                } else if (!selectedConnection.value && connections.value.length) {
                    selectedConnection.value = connections.value[0].id;
                }
                updateInstitutionForSelected();
            } catch (err) {
                console.error('No se pudo obtener el estado de Plaid', err);
                connected.value = false;
            }
        };

        const handleTransactionsResponse = (data) => {
            if (!data || typeof data !== 'object') {
                toastr.error('No se pudo leer la respuesta de transacciones. Intenta nuevamente.');
                return false;
            }
            if (data.error_code === 'ITEM_LOGIN_REQUIRED') {
                toastr.error('Tu banco requiere que vuelvas a conectarte. Presiona "Connect bank account" para actualizar el acceso.');
                connected.value = false;
                return false;
            }
            if (data.error || data.error_message || data.error_type) {
                const msg = data.error_message || data.display_message || data.error || 'Error al obtener transacciones.';
                toastr.error(msg);
                return false;
            }
            if (!Array.isArray(data.transactions)) {
                toastr.error('No se recibieron transacciones en la respuesta.');
                return false;
            }
            return true;
        };

        const getTransactions = async () => {
            loading.value = true;
            updateInstitutionForSelected();
            try {
                const response = await fetch(`${appUrl}api/plaid/transactions`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ client_id: cid, connection_id: selectedConnection.value })
                });
                const data = await response.json();
                if (!handleTransactionsResponse(data)) return;
                transactions.value = data.transactions || [];
                accounts.value = data.accounts || [];
                ensurePageInRange();
            } catch (err) {
                console.error('Fallo al obtener transacciones', err);
                toastr.error('No se pudieron obtener transacciones. Intenta nuevamente.');
            } finally {
                loading.value = false;
            }
        };

        const connect = async () => {
            loading.value = true;
            try {
                const res = await fetch(`${appUrl}api/plaid/create_link_token${cid ? '?client_id=' + encodeURIComponent(cid) : ''}`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' }
                });
                if (!res.ok) {
                    const err = await res.json().catch(() => ({}));
                    toastr.error('Error obteniendo link token: ' + (err.error || res.status));
                    return;
                }
                const body = await res.json();
                const linkToken = body.link_token;
                if (!linkToken) {
                    toastr.error('Link token invalido.');
                    return;
                }

                const handler = Plaid.create({
                    token: linkToken,
                    onSuccess: async function(public_token) {
                        try {
                            const exchangeRes = await fetch(`${appUrl}api/plaid/exchange_public_token`, {
                                method: 'POST',
                                headers: { 'Content-Type': 'application/json' },
                                body: JSON.stringify({ public_token, client_id: cid })
                            });
                            const exchangeBody = await exchangeRes.json();
                            if (!exchangeRes.ok) {
                                toastr.error(exchangeBody.error || 'Connection error');
                                return;
                            }
                            toastr.success('Connected account');
                            await fetchStatus();
                            if (connected.value) {
                                getTransactions();
                            }
                        } catch (err) {
                            console.error(err);
                            toastr.error('Connection error');
                        }
                    },
                    onExit: function(err) {
                        if (err) {
                            console.error('Plaid Link exit with error:', err);
                        }
                    },
                    onEvent: function(eventName, metadata) {
                        console.log('Plaid event', eventName, metadata);
                    }
                });
                handler.open();
            } catch (err) {
                console.error(err);
                toastr.error('Plaid Link error: ' + err.message);
            } finally {
                loading.value = false;
            }
        };

        onMounted(async () => {
            await fetchStatus();
            if (connected.value) {
                getTransactions();
            }
        });

        return {
            loading,
            connected,
            institution,
            connections,
            selectedConnection,
            transactions,
            search,
            filtered,
            paged,
            pageSize,
            currentPage,
            totalPages,
            pageStart,
            toggleSort,
            goToPage,
            pageNumbers,
            sortKey,
            sortDir,
            connect,
            getTransactions,
            accountName,
            accountSubtype
        };
    }
}).mount('#plaid-app');
})();
{/literal}
</script>
{/block}
